<?php
/**
 * @package		EasyBlog
 * @copyright	Copyright (C) 2010 Stack Ideas Private Limited. All rights reserved.
 * @license		GNU/GPL, see LICENSE.php
 *
 * EasyBlog is free software. This version may have been modified pursuant
 * to the GNU General Public License, and as distributed it includes or
 * is derivative of works licensed under the GNU General Public License or
 * other free or open source software licenses.
 * See COPYRIGHT.php for copyright notices and details.
 */

defined('_JEXEC') or die('Restricted access');

/*
 * Comment utilities class.
 *
 */
require_once( EBLOG_HELPERS . DS . 'helper.php' );

class EasyBlogCommentHelper
{
	public $pagination = null;

	/**
	 * add notification email into mailq
	 *
	 * @param	object comment jtable
	 **/
	public static function addNotification($comment, $skipAdmin = false, $parentId = 0)
	{
		$config = EasyBlogHelper::getConfig();

		//get blog title
		$blog	= EasyBlogHelper::getTable( 'Blog', 'Table' );
		$blog->load($comment->post_id);

		$teamIdLink = '';
		if( isset( $blog->team ) )
		{
			$teamIdLink = '&team=' . $blog->team;
		}


		$posterName = '';
		if($comment->created_by != 0)
		{

			$creator	= EasyBlogHelper::getTable( 'Profile', 'Table' );
			$user   = JFactory::getUser($comment->created_by);
			$creator->setUser($user);

			$posterName = $creator->getName();
		}
		else
		{
			$posterName = $comment->name;
		}

		$emailData['blogTitle']		= $blog->title;
		$emailData['commentTitle']	= '';
		if( $config->get('comment_show_title', 0) )
		{
			$emailData['commentTitle']	= ( empty($comment->title) ) ? '-' : $comment->title;
		}

		$emailData['comment']		= $comment->comment;
		$emailData['commentPoster']	= $posterName;
		$emailData['commentLink']	= EasyBlogRouter::getRoutedURL( 'index.php?option=com_easyblog&view=entry'.$teamIdLink.'&id='. $blog->id, false, true) . '#comment-' . $comment->id;


		//send email to admin and blogger
		$notify		= EasyBlogHelper::getNotification();
		$toEmails	= array();

		if(! $skipAdmin)
		{
			if($config->get('notification_commentadmin'))
			{

				if($config->get('notification_email', '') != '')
				{
					$toNotifyEmails	= explode(',', $config->get('notification_email'));
					$notify->sendEmails($toNotifyEmails, JText::_('COM_EASYBLOG_NEW_COMMENT_ADDED'), '', 'email.comment.new.php', $emailData);
				}
				else
				{
					$admins	= $notify->getAdmins();
					foreach($admins as $admin)
					{
						$toEmails[] = $admin->id;
					}
				}
			}
		}

		if( empty( $parentId ) )
		{
			// send notification to blog authoer only on new comment
			if($config->get('notification_commentauthor'))
			{
				//blogger
				$toEmails[] = $blog->created_by;
			}
		}

		if(! empty($toEmails))
		{
			$notify->send('', $toEmails, JText::_('COM_EASYBLOG_NEW_COMMENT_ADDED'), '', 'email.comment.new.php', $emailData);
		}


		// send notification to subcribers
		if( empty( $parentId ) )
		{
			if( ($config->get('main_subscription') && $blog->subscription == '1') && $config->get('notification_commentsubscriber'))
			{
				$subscribers = $blog->getSubscribers( array( $comment->email ) );

				$emails = array();
				if(! empty($subscribers))
				{
					require_once( EBLOG_HELPERS . DS . 'subscription.php' );
					$notify	= EasyBlogHelper::getNotification();

					foreach($subscribers as $subscriber)
					{
						$subscriber->type	= 'subscription';
						$unsublink = EasyBlogSubscription::getUnsubscribeLink($subscriber, true);
						$emailData['unsubscribeLink'] = JText::sprintf('EMAIL_UNSUBSCRIBE_INFO' , '<a href="'.$unsublink.'">'.JText::_('COM_EASYBLOG_URL_HERE').'</a>');

						$notify->sendEmails($subscriber->email, JText::sprintf('COM_EASYBLOG_NEW_COMMENT_ADDED_IN_BLOG_ENTRY', $blog->title), '', 'email.subscription.comment.new.php', $emailData);
					}
				}
			}
		}
		else
		{
			// original commentor that the reply posted to.
			$parentCommt	= EasyBlogHelper::getTable( 'Comment', 'Table' );
			$parentCommt->load($parentId);

			$commentAuthorEmail = $parentCommt->email;

			if( !empty($commentAuthorEmail) )
			{
				$notify->sendEmails($commentAuthorEmail, JText::_('COM_EASYBLOG_NEW_COMMENT_ADDED'), '', 'email.comment.new.php', $emailData);
			}
		}

	}

	/**
	 * Format comments data
	 **/
	public static function format( $comments )
	{

		for($i = 0; $i < count($comments); $i++)
		{
			$row 			= $comments[$i];
			$user 			= EasyBlogHelper::getTable( 'Profile', 'Table' );
			$row->poster    = $user->load( $row->created_by );
			$row->comment   = nl2br($row->comment);
		}

		return $comments;
	}

	/*
	 * Determines whether the current comment system used is built in or an external tool
	 *
	 * @param	null
	 * @return	boolean	True if built in false otherwise.
	 */
	public static function isBuiltin()
	{
		jimport( 'joomla.filesystem.file' );

		$config		= EasyBlogHelper::getConfig();

		if( !$config->get( 'main_comment' ) )
		{
			return false;
		}

		if( $config->get( 'intensedebate' ) )
		{
			return false;
		}

		if( $config->get( 'comment_disqus' ) )
		{
			return false;
		}

		$file	= JPATH_ROOT . DS . 'components' . DS . 'com_jomcomment' . DS . 'jomcomment.php';

		if( $config->get( 'comment_jomcomment' ) && JFile::exists( $file ) )
		{
			return false;
		}

		$file	= JPATH_ROOT . DS . 'components' . DS . 'com_jcomments' . DS . 'jcomments.php';

		if( $config->get( 'comment_jcomments' ) && JFile::exists( $file ) )
		{
			return false;
		}

		$file	= JPATH_ROOT . DS . 'components' . DS . 'com_rscomments' . DS . 'rscomments.php';

		if( $config->get( 'comment_rscomments' ) && JFile::exists( $file ) )
		{
			return false;
		}

		if( $config->get( 'comment_facebook' ) )
		{
			return false;
		}

		return true;
	}

	/**
	 * Retrieves the comment count for the specific blog
	 *
	 * @param	int	$blogId	The blog id.
	 **/
	public static function getCommentCount( $blogId )
	{
		$config		= EasyBlogHelper::getConfig();


		if( $config->get( 'comment_compojoom' ) )
		{
			$file	= JPATH_ROOT . DS . 'administrator' . DS . 'components' . DS . 'com_comment' . DS . 'plugin' . DS . 'com_easyblog' . DS . 'josc_com_easyblog.php';

			if( JFile::exists( $file ) )
			{
				require_once( $file );
				$blog		= EasyBlogHelper::getTable( 'Blog' );
				$blog->load( $blogId );
				return CommentEasyBlog::output( $blog , array() , true );
			}
		}

		if( $config->get( 'intensedebate' ) )
		{
			return false;
		}

		if( $config->get( 'comment_disqus' ) )
		{
			$string = '<!-- Disqus -->';
			$string .= '<div class="catItemCommentsLink">';
			$string .= '<a href="' . EasyBlogRouter::_('index.php?option=com_easyblog&view=entry&id='.$blogId) . '#disqus_thread">Comments</a>';
			$string .= '</div>';

			return $string;
			// return false;
		}

		if( $config->get( 'comment_jomcomment' ) )
		{
			$file 	= JPATH_ROOT . DS . 'components' . DS . 'com_jomcomment' . DS . 'helper' . DS . 'minimal.helper.php';

			jimport( 'joomla.filesystem.file' );

			if( !JFile::exists( $file ) )
			{
				return false;
			}

			require_once( $file );

			return jcCountComment( $blogId , 'com_easyblog' );
		}

		if( $config->get( 'comment_jcomments' ) )
		{
			$db 	= JFactory::getDBO();
			$query 	= 'SELECT COUNT(1) FROM ' . $db->nameQuote( '#__jcomments' ) . ' '
					. 'WHERE ' . $db->nameQuote( 'object_id' ) . '=' . $db->Quote( $blogId ) . ' '
					. 'AND ' . $db->nameQuote( 'object_group' ) . '=' . $db->Quote( 'com_easyblog' ) . ' '
					. 'AND ' . $db->nameQuote( 'published' ) . '=' . $db->Quote( 1 );
			$db->setQuery( $query );

			$total 	= $db->loadResult();

			return $total;
		}

		if( $config->get( 'comment_rscomments' ) )
		{
			return false;
		}

		if( $config->get( 'comment_facebook' ) )
		{
			return false;
		}

		$db		= JFactory::getDBO();
		$query	= 'SELECT COUNT(1) FROM '
				. $db->nameQuote( '#__easyblog_comment' )
				. ' WHERE ' . $db->nameQuote( 'post_id' ) . '=' . $db->Quote( $blogId )
				. ' AND `published` = ' . $db->Quote('1');

		$db->setQuery( $query );

		$count	= $db->loadResult();

		return $count;
	}

	public static function getBlogCommentLite(  $blogId, $limistFrontEnd = 0, $sort = 'asc')
	{
		return EasyBlogCommentHelper::getBlogComment($blogId, $limistFrontEnd, $sort, true);
	}

	public function getBlogComment(  $blogId, $limistFrontEnd = 0, $sort = 'asc', $isLite = false)
	{
		$config			= EasyBlogHelper::getConfig();
		$comments       = array();

		if( self::isBuiltin() )
		{
			$model  			= JModel::getInstance( 'Blog', 'EasyBlogModel' );
			$comments   		= $model->getBlogComment( $blogId , $limistFrontEnd , $sort, $isLite);
			$this->pagination	= $model->getPagination();
		}

		return $comments;
	}

	public static function getCommentHTML( $blog , $comments = array() , $pagination = '' )
	{
		$config			= EasyBlogHelper::getConfig();
		$path			= EBLOG_CLASSES . DS . 'comments';
		$registration	= $config->get( 'comment_registeroncomment' );

		// Double check this with Joomla's registration component
		if( $registration )
		{
			$params			= JComponentHelper::getParams( 'com_users' );
			$registration	= $params->get( 'allowUserRegistration' ) == '0' ? false : $registration;
		}

		if( !$blog->allowcomment )
		{
			$theme	= new CodeThemes();
			$theme->set( 'allowComment'	, 0 );
			$theme->set( 'config'		, $config );
			$theme->set( 'canRegister'	, $registration );
			$theme->set( 'blog'			, $blog );
			return $theme->fetch( 'blog.comment.box.php' );
		}

		if( $config->get( 'comment_facebook' ) )
		{
			require_once( $path . DS . 'facebook.php' );

			return EasyBlogCommentFacebook::getHTML( $blog );
		}

		if( $config->get( 'comment_compojoom' ) )
		{
			$file	= JPATH_ROOT . DS . 'administrator' . DS . 'components' . DS . 'com_comment' . DS . 'plugin' . DS . 'com_easyblog' . DS . 'josc_com_easyblog.php';

			if( JFile::exists( $file ) )
			{
				require_once( $file );
				return CommentEasyBlog::output( $blog , array() );
			}
		}

		if( $config->get('comment_intensedebate') )
		{
			require_once( $path . DS . 'intensedebate.php' );

			return EasyBlogCommentIntenseDebate::getHTML( $blog );
		}

		if( $config->get('comment_disqus') )
		{
			require_once( $path . DS . 'disqus.php' );

			return EasyBlogCommentDisqus::getHTML( $blog );
		}

		if( $config->get('comment_jomcomment') )
		{
			$file	= JPATH_ROOT . DS . 'components' . DS . 'com_jomcomment' . DS . 'jomcomment.php';

			// Test if jomcomment exists.
			if( JFile::exists( $file ) )
			{
				require_once( $path . DS . 'jomcomment.php' );

				return EasyBlogCommentJomComment::getHTML( $blog );
			}
		}

		if( $config->get('comment_jcomments' ) )
		{
			$file	= JPATH_ROOT . DS . 'components' . DS . 'com_jcomments' . DS . 'jcomments.php';

			if( JFile::exists( $file ) )
			{
				require_once( $path . DS . 'jcomments.php' );
				return EasyBlogCommentJComments::getHTML( $blog );
			}
		}

		if($config->get('comment_rscomments') )
		{
			$file	= JPATH_ROOT . DS . 'components' . DS . 'com_rscomments' . DS . 'rscomments.php';

			if( JFile::exists( $file ) )
			{
				include_once( $path . DS . 'rscomments.php' );
				return EasyBlogCommentRSComments::getHTML( $blog );
			}
		}

		//check if bbcode enabled or not.
		if($config->get('comment_bbcode'))
		{
			EasyBlogCommentHelper::loadBBCode();
		}

		// If all else fail, try to use the default comment system
		$theme		= new CodeThemes();

		// setup my own info to show in comment form area
		$my			= JFactory::getUser();
		$profile	= EasyBlogHelper::getTable( 'Profile', 'Table' );
		$profile->load( $my->id );

		$my->avatar 		= $profile->getAvatar();
		$my->displayName    = $profile->getName();
		$my->url			= $profile->url;
		$blogURL			= base64_encode( EasyBlogRouter::_('index.php?option=com_easyblog&view=entry&id=' . $blog->id , false ) );
		$loginURL			= ( EasyBlogHelper::getJoomlaVersion() >= 1.6) ? EasyBlogRouter::_('index.php?option=com_users&view=login&return=' . $blogURL) : EasyBlogRouter::_('index.php?option=com_user&view=login&return=' . $blogURL);
		$enableRecaptcha	= $config->get('comment_recaptcha');
		$publicKey			= $config->get('comment_recaptcha_public');

		// check if the user has subcribed to this thread
		$subscriptionId		= false;
		if ($my->id > 0)
		{
			$blogModel		= EasyblogHelper::getModel('Blog');
			$subscriptionId	= $blogModel->isBlogSubscribedUser( $blog->id , $my->id , $my->email );
			$subscriptionId	= is_null($subscriptionId) ? false : $subscriptionId;
		}

		$theme->set('loginURL' 		, $loginURL );
		$theme->set('blog'			, $blog );
		$theme->set('my'			, $my );
		$theme->set('config'		, $config );
		$theme->set('blogComments'	, $comments );
		$theme->set('pagination'	, $pagination );
		$theme->set('allowComment'	, true );
		$theme->set('canRegister'	, $registration );
		$theme->set('acl'			, EasyBlogACLHelper::getRuleSet() );
		$theme->set('subscriptionId'	, $subscriptionId);

		return $theme->fetch( 'blog.comment.box.php' );
	}

	public static function loadBBCode()
	{
		$document 	= JFactory::getDocument();
		$out   = '<link rel="stylesheet" type="text/css" href="'.rtrim(JURI::root(), '/').'/components/com_easyblog/classes/markitup/skins/simple/style.css" />' . "\n";
		$out  .= '<link rel="stylesheet" type="text/css" href="'.rtrim(JURI::root(), '/').'/components/com_easyblog/classes/markitup/sets/bbcode/style.css" />' . "\n";

		$out  .= '<script type="text/javascript" src="'.rtrim(JURI::root(), '/').'/components/com_easyblog/classes/markitup/jquery.markitup.pack.js"></script>' . "\n";

		$bold		= JText::_( 'COM_EASYBLOG_BBCODE_BOLD' );
		$italic		= JText::_( 'COM_EASYBLOG_BBCODE_ITALIC' );
		$underline	= JText::_( 'COM_EASYBLOG_BBCODE_UNDERLINE' );
		$picture	= JText::_( 'COM_EASYBLOG_BBCODE_PICTURE' );
		$bullet		= JText::_( 'COM_EASYBLOG_BBCODE_BULLETS' );
		$numeric	= JText::_( 'COM_EASYBLOG_BBCODE_NUMERIC' );
		$list		= JText::_( 'COM_EASYBLOG_BBCODE_LIST' );
		$quote		= JText::_( 'COM_EASYBLOG_BBCODE_QUOTES' );
		$clean		= JText::_( 'COM_EASYBLOG_BBCODE_CLEAN' );
		$happy		= JText::_( 'COM_EASYBLOG_BBCODE_HAPPY' );
		$smile		= JText::_( 'COM_EASYBLOG_BBCODE_SMILE' );
		$surprised	= JText::_( 'COM_EASYBLOG_BBCODE_SURPRISED' );
		$tongue		= JText::_( 'COM_EASYBLOG_BBCODE_TONGUE' );
		$unhappy	= JText::_( 'COM_EASYBLOG_BBCODE_UNHAPPY' );
		$wink		= JText::_( 'COM_EASYBLOG_BBCODE_WINK' );

		$bbcode	=<<<EOF
EasyBlogBBCodeSettings = {

	previewParserVar: 'data',
	markupSet: [
		{name:'$bold', key:'B', openWith:'[b]', closeWith:'[/b]'},
		{name:'$italic', key:'I', openWith:'[i]', closeWith:'[/i]'},
		{name:'$underline', key:'U', openWith:'[u]', closeWith:'[/u]'},
		{separator:'---------------' },
		{name:'$picture', key:'P', replaceWith:'[img][![Url]!][/img]'},
		{separator:'---------------' },
		{name:'$bullet', openWith:'[list]\\n', closeWith:'\\n[/list]'},
		{name:'$numeric', openWith:'[list=[![Starting number]!]]\\n', closeWith:'\\n[/list]'},
		{name:'$list', openWith:'[*] '},
		{separator:'---------------' },
		{name:'$quote', openWith:'[quote]', closeWith:'[/quote]'},
		{name:'$clean', className:"clean", replaceWith:function(markitup) { return markitup.selection.replace(/\[(.*?)\]/g, "") } },
		{separator:'---------------' },
		{name:'$happy', openWith:':)'},
		{name:'$smile', openWith:':D'},
		{name:'$surprised', openWith:':o'},
		{name:'$tongue', openWith:':p'},
		{name:'$unhappy', openWith:':('},
		{name:'$wink', openWith:';)'}
	]
};

EOF;

		$out 	 .= '<script type="text/javascript">' . "\n";
		$out	.= $bbcode;
		$out  	.= 'sQuery(document).ready(function() {' . "\n";
		$out  	.= '	sQuery("#comment").markItUp(EasyBlogBBCodeSettings);' . "\n";
		$out  	.= '});' . "\n";
		$out  	.= '</script>' . "\n";
		$document->addCustomTag($out);
	}

	function parseBBCode($text)
	{
		//$text	= htmlspecialchars($text , ENT_NOQUOTES );
		$text	= trim($text);
		//$text   = nl2br( $text );

		//$text = preg_replace_callback('/\[code\](.*?)\[\/code\]/ms', "escape", $text);
		$text = preg_replace_callback('/\[code( type="(.*?)")?\](.*?)\[\/code\]/ms', 'escape' , $text );

		// BBCode to find...
		$in = array( 	 '/\[b\](.*?)\[\/b\]/ms',
						 '/\[i\](.*?)\[\/i\]/ms',
						 '/\[u\](.*?)\[\/u\]/ms',
						 '/\[img\](.*?)\[\/img\]/ms',
						 '/\[email\](.*?)\[\/email\]/ms',
						 '/\[url\="?(.*?)"?\](.*?)\[\/url\]/ms',
						 '/\[size\="?(.*?)"?\](.*?)\[\/size\]/ms',
						 '/\[color\="?(.*?)"?\](.*?)\[\/color\]/ms',
						 '/\[quote](.*?)\[\/quote\]/ms',
						 '/\[list\=(.*?)\](.*?)\[\/list\]/ms',
						 '/\[list\](.*?)\[\/list\]/ms',
						 '/\[\*\]\s?(.*?)\n/ms'
		);
		// And replace them by...
		$out = array(	 '<strong>\1</strong>',
						 '<em>\1</em>',
						 '<u>\1</u>',
						 '<img src="\1" alt="\1" />',
						 '<a href="mailto:\1">\1</a>',
						 '<a href="\1">\2</a>',
						 '<span style="font-size:\1%">\2</span>',
						 '<span style="color:\1">\2</span>',
						 '<blockquote>\1</blockquote>',
						 '<ol start="\1">\2</ol>',
						 '<ul>\1</ul>',
						 '<li>\1</li>'
		);

		$tmp    = preg_replace( $in , '' , $text );
		$text	= EasyBlogCommentHelper::replaceURL( $tmp, $text );
		$text	= preg_replace($in, $out, $text);

		// Smileys to find...
		$in = array( 	 ':)',
						 ':D',
						 ':o',
						 ':p',
						 ':(',
						 ';)'
		);

		// And replace them by...
		$out = array(	 '<img alt=":)" src="'.EBLOG_EMOTICONS_DIR.'emoticon-happy.png" />',
						 '<img alt=":D" src="'.EBLOG_EMOTICONS_DIR.'emoticon-smile.png" />',
						 '<img alt=":o" src="'.EBLOG_EMOTICONS_DIR.'emoticon-surprised.png" />',
						 '<img alt=":p" src="'.EBLOG_EMOTICONS_DIR.'emoticon-tongue.png" />',
						 '<img alt=":(" src="'.EBLOG_EMOTICONS_DIR.'emoticon-unhappy.png" />',
						 '<img alt=";)" src="'.EBLOG_EMOTICONS_DIR.'emoticon-wink.png" />'
		);
		$text = str_replace($in, $out, $text);

		// paragraphs
		$text = str_replace("\r", "", $text);
		$text = "<p>".preg_replace("/(\n){2,}/", "</p><p>", $text)."</p>";


		$text = preg_replace_callback('/<pre>(.*?)<\/pre>/ms', "removeBr", $text);
		$text = preg_replace('/<p><pre>(.*?)<\/pre><\/p>/ms', "<pre>\\1</pre>", $text);

		$text = preg_replace_callback('/<ul>(.*?)<\/ul>/ms', "removeBr", $text);
		$text = preg_replace('/<p><ul>(.*?)<\/ul><\/p>/ms', "<ul>\\1</ul>", $text);

		return $text;
	}

	public static function replaceURL( $tmp , $text )
	{
		$pattern    = '@(https?://[-\w\.]+(:\d+)?(/([\w/_\.-]*(\?\S+)?)?)?)@';
		preg_match_all( $pattern , $tmp , $matches );

		if( isset( $matches[ 0 ] ) && is_array( $matches[ 0 ] ) )
		{
			foreach( $matches[ 0 ] as $match )
			{
				$match	= str_ireplace( array( '<br' , '<br />' ) , '' , $match );
				$match 	= '@(?<!\])(' . $match . ')@';
				$text 	= preg_replace( $match , '<a href="\1">\1</a>' , $text );
				// $text   = str_ireplace( $match , '<a href="' . $match . '">' . $match . '</a>' , $text );
			}
		}

		$text   = str_ireplace( '&quot;' , '"', $text );

		return $text;
	}
}

// clean some tags to remain strict
// not very elegant, but it works. No time to do better ;)
if (!function_exists('removeBr')) {
	function removeBr($s) {
		return str_replace("<br />", "", $s[0]);
	}
}

// BBCode [code]
if (!function_exists('escape')) {
	function escape($s) {
		global $text;
		$text = strip_tags($text);
		$code = $s[1];
		$code = htmlspecialchars($code);
		$code = str_replace("[", "&#91;", $code);
		$code = str_replace("]", "&#93;", $code);
		return '<pre><code>'.$code.'</code></pre>';
	}
}
