<?php
/**
 * @package		EasyBlog
 * @copyright	Copyright (C) 2010 Stack Ideas Private Limited. All rights reserved.
 * @license		GNU/GPL, see LICENSE.php
 *
 * EasyBlog is free software. This version may have been modified pursuant
 * to the GNU General Public License, and as distributed it includes or
 * is derivative of works licensed under the GNU General Public License or
 * other free or open source software licenses.
 * See COPYRIGHT.php for copyright notices and details.
 */

defined('_JEXEC') or die('Restricted access');

jimport('joomla.filesystem.file' );
jimport('joomla.filesystem.folder' );
jimport('joomla.html.parameter' );
jimport('joomla.application.component.model');
jimport('joomla.access.access');

require_once( JPATH_ROOT.DS.'components'.DS.'com_easyblog'.DS.'constants.php' );
require_once( EBLOG_CLASSES . DS . 'themes.php' );
require_once( EBLOG_CLASSES . DS . 'adsense.php' );
require_once( EBLOG_HELPERS . DS . 'image.php' );
require_once( EBLOG_HELPERS . DS . 'router.php' );
require_once( EBLOG_HELPERS . DS . 'date.php' );
require_once( EBLOG_HELPERS . DS . 'integrate.php' );
require_once( EBLOG_HELPERS . DS . 'acl.php' );
require_once( EBLOG_HELPERS . DS . 'comment.php' );
require_once( EBLOG_HELPERS . DS . 'socialshare.php' );

class EasyBlogHelper
{
	/**
	 * Retrieve specific helper objects.
	 *
	 * @param	string	$helper	The helper class . Class name should be the same name as the file. e.g EasyBlogXXXHelper
	 * @return	object	Helper object.
	 **/
	public static function getHelper( $helper )
	{
		static $obj	= array();

		if( !isset( $obj[ $helper ] ) )
		{
			$file	= EBLOG_HELPERS . DS . JString::strtolower( $helper ) . '.php';

			if( JFile::exists( $file ) )
			{
				require_once( $file );
				$class	= 'EasyBlog' . ucfirst( $helper ) . 'Helper';

				$obj[ $helper ]	= new $class();
			}
			else
			{
				$obj[ $helper ]	= false;
			}
		}

		return $obj[ $helper ];
	}

	public static function addJomsocialPoint( $action , $userId = 0 )
	{
		$my			= JFactory::getUser();

		if( !empty( $userId ) )
		{
			$my			= JFactory::getUser( $userId );
		}

		$config		= EasyBlogHelper::getConfig();

		if( $my->id != 0 && $config->get('main_jomsocial_userpoint') )
		{
			$jsUserPoint	= JPATH_ROOT . DS . 'components' . DS . 'com_community' . DS . 'libraries' . DS . 'userpoints.php';

			if( JFile::exists( $jsUserPoint ) )
			{
				require_once( $jsUserPoint );
				CUserPoints::assignPoint( $action , $my->id );
			}
		}
		return true;
	}

	public static function addJomsocialActivity( $blog , $command , $title , $actor , $contents )
	{
		$jsCoreFile	= JPATH_ROOT . DS . 'components' . DS . 'com_community' . DS . 'libraries' . DS . 'core.php';
		$config		= EasyBlogHelper::getConfig();
		$lang		= JFactory::getLanguage();
		$lang->load( 'com_easyblog' , JPATH_ROOT );

		if( !JFile::exists( $jsCoreFile ) )
		{
			return false;
		}
		require_once( $jsCoreFile );

		$obj			= new stdClass();
		$obj->title		= $title;
		$obj->content	= $contents;
		$obj->cmd 		= $command;

		if( $config->get( 'integrations_jomsocial_activity_likes' ) )
		{
			$obj->like_id   = $blog->id;
			$obj->like_type = 'com_easyblog';
		}

		if( $config->get( 'integrations_jomsocial_activity_comments' ) )
		{
			$obj->comment_id    = $blog->id;
			$obj->comment_type  = 'com_easyblog';
		}

		$obj->actor   	= $actor;
		$obj->target  	= 0;
		$obj->app		= 'com_easyblog';
		$obj->cid		= $blog->id;

		// add JomSocial activities
		CFactory::load ( 'libraries', 'activities' );
		CActivityStream::add($obj);
	}

	/*
	 * type - string - info | warning | error
	 */

	public static function setMessageQueue($message, $type = 'info')
	{
		$session 	= JFactory::getSession();

		$msgObj = new stdClass();
		$msgObj->message    = $message;
		$msgObj->type       = strtolower($type);

		//save messsage into session
		$session->set('eblog.message.queue', $msgObj, 'EBLOG.MESSAGE');

	}

	public static function getMessageQueue()
	{
		$session 	= JFactory::getSession();
		$msgObj 	= $session->get('eblog.message.queue', null, 'EBLOG.MESSAGE');

		//clear messsage into session
		$session->set('eblog.message.queue', null, 'EBLOG.MESSAGE');

		return $msgObj;
	}

	public static function getPermalink( $title )
	{
		$permalink	= EasyBlogRouter::generatePermalink( $title );

		// Make sure no such permalink exists.
		$originalSlug   = $permalink;
		$i				= 1;
		while( EasyBlogRouter::_isBlogPermalinkExists( $permalink ) )
		{
			$permalink	= $originalSlug . '-' . $i;
			$i++;
		}

		return $permalink;
	}

	public static function getNotification()
	{
		static $notify = false;

		if( !$notify )
		{
			require_once( EBLOG_CLASSES . DS . 'notification.php' );
			$notify	= new ENotification();
		}
		return $notify;

	}

	public static function getMailQueue()
	{
		static $mailq = false;

		if( !$mailq )
		{
			require_once( EBLOG_CLASSES . DS . 'mailqueue.php' );

			$mailq	= new EMailQueue();
		}
		return $mailq;

	}

	public static function getRegistor()
	{
		static $registor = false;

		if( !$registor )
		{
			require_once( EBLOG_CLASSES . DS . 'registration.php' );

			$registor	= new ERegistration();
		}
		return $registor;

	}

	private static function getParser()
	{
		$parser		= JFactory::getXMLParser('Simple');

		$data		= new stdClass();

		// Get the xml file
		$site		= EBLOG_UPDATES_SERVER;
		$xml		= 'stackideas.xml';
		$contents	= '';

		$handle		= @fsockopen( $site , 80, $errno, $errstr, 30);

		if( !$handle )
			return false;

		$out = "GET /$xml HTTP/1.1\r\n";
		$out .= "Host: $site\r\n";
		$out .= "Connection: Close\r\n\r\n";

		fwrite($handle, $out);

		$body		= false;

		while( !feof( $handle ) )
		{
			$return	= fgets( $handle , 1024 );

			if( $body )
			{
				$contents	.= $return;
			}

			if( $return == "\r\n" )
			{
				$body	= true;
			}
		}
		fclose($handle);

		$parser->loadString( $contents );

		return $parser;
	}

	private static function getLocalParser()
	{
		$parser		= JFactory::getXMLParser('Simple');

		$data		= new stdClass();

		$contents	= JFile::read( JPATH_ROOT . DS . 'administrator' . DS . 'components' . DS . 'com_easyblog' . DS . 'easyblog.xml' );
		$parser->loadString( $contents );

		return $parser;
	}

	public static function getLocalVersion( $buildOnly = false )
	{
		$parser	= EasyBlogHelper::getLocalParser();

		if( !$parser )
			return false;

		$element	= $parser->document->getElementByPath( 'version' );

		$data		= $element->data();
		if( $buildOnly )
		{
			$data	= explode( '.' , $data );
			return $data[2];
		}
		return $data;
	}

	public static function getVersion()
	{
		$parser	= EasyBlogHelper::getParser();

		if( !$parser )
			return false;

		$element	= $parser->document->getElementByPath( 'easyblog/version' );

		return $element->data();
	}

	public static function getRecentNews()
	{
		$parser	= EasyBlogHelper::getParser();

		if( !$parser )
			return false;

		$items	= $parser->document->getElementByPath('easyblog/news');

		$news	= array();

		foreach($items->children() as $item)
		{
			$element	= $item->getElementByPath( 'title' );
			$obj		= new stdClass();
			$obj->title	= $element->data();
			$element	= $item->getElementByPath( 'description' );
			$obj->desc	= $element->data();
			$element	= $item->getElementByPath( 'pubdate' );
			$obj->date	= $element->data();
			$news[]		= $obj;
		}

		return $news;
	}

	public static function getTable( $tableName , $prefix = 'Table' )
	{
		JTable::addIncludePath( EBLOG_TABLES );

		$tbl    = JTable::getInstance( $tableName , $prefix );
		return $tbl;
	}

	public static function getModel( $name )
	{
		static $model = array();

		if( !isset( $model[ $name ] ) )
		{
			$file	= JString::strtolower( $name );
			$path	= JPATH_ROOT . DS . 'components' . DS . 'com_easyblog' . DS . 'models' . DS . $file . '.php';

			jimport('joomla.filesystem.path');
			if ( JFolder::exists( $path ))
			{
				JError::raiseWarning( 0, 'Model file not found.' );
			}

			$modelClass		= 'EasyBlogModel' . ucfirst( $name );

			if( !class_exists( $modelClass ) )
				require_once( $path );


			$model[ $name ] = new $modelClass();
		}

		return $model[ $name ];
	}

	public static function getConfig()
	{
		static $config	= null;

		if( is_null( $config ) )
		{
			//load default ini data first
			$ini		= JPATH_ROOT . DS . 'administrator' . DS . 'components' . DS . 'com_easyblog' . DS . 'configuration.ini';
			$raw		= JFile::read($ini);
			$config	= new JParameter($raw);

			//get config stored in db
			JTable::addIncludePath( JPATH_ROOT . DS . 'administrator' . DS . 'components' . DS . 'com_easyblog' . DS . 'tables' );
			$dbConfig	= EasyBlogHelper::getTable( 'configs' , 'Table' );
			$dbConfig->load( 'config' );

			if( EasyBlogHelper::getJoomlaVersion() >= '1.7' )
			{
				$config->bind( $dbConfig->params , 'INI' );
			}
			else
			{
				$config->bind( $dbConfig->params );
			}
		}

		return $config;
	}

	/*
	 * Method used to determine whether the user a guest or logged in user.
	 * return : boolean
	 */
	public static function isLoggedIn()
	{
		$my			= JFactory::getUser();
		$loggedIn	= (empty($my) || $my->id == 0) ? false : true;

		return $loggedIn;
	}

	public static function isSiteAdmin( $userId = null )
	{
		$my	= JFactory::getUser( $userId );

		$admin  = false;
		if(EasyBlogHelper::getJoomlaVersion() >= '1.6')
		{
			$admin	= $my->authorise('core.admin');
		}
		else
		{
			$admin	= $my->usertype == 'Super Administrator' || $my->usertype == 'Administrator' ? true : false;
		}
		return $admin;
	}

	public static function isTeamAdmin()
	{
		static $isAdmin	= null;

		$my	= JFactory::getUser();

		if($my->id == 0)
		{
			return false;
		}
		else
		{
			if( !isset( $isAdmin[ $my->id ] ) )
			{
				$model			= self::getModel( 'TeamBlogs' );

				$isTeamAdmin	= $model->checkIsTeamAdmin($my->id);

				if($isTeamAdmin === false)
				{
					$isTeamAdmin    =EasyBlogHelper::isSiteAdmin();
				}

				$isAdmin[$my->id]   = $isTeamAdmin;
			}

			return $isAdmin[$my->id];
		}
	}

	/*
	 * Method used to determine the current user is the owner of a blog post.
	 * return : boolean
	 */
	public static function isMineBlog($userId1, $userId2)
	{
		return ($userId1 == $userId2) && (($userId1 != 0) || ($userId2 != 0) );
	}

	/**
	 * Gets the comment responses for an article
	 * Deprecated since 2.0
	 **/
	public static function getCommentCount( $articleId )
	{
		return EasyBlogHelper::getHelper( 'Comment' )->getCommentCount( $articleId );
	}

	public static function formatTeamMembers( $data )
	{
		$memberInfo = array();
		for($i = 0; $i < count($data); $i++)
		{
			$member     = $data[$i];

			$profile 	= EasyBlogHelper::getTable( 'Profile', 'Table' );
			$profile->load( $member->user_id );
			$profile->displayName	= $profile->getName();
			$memberInfo[]   = $profile;
		}

		return $memberInfo;
	}

	public static function removeFeaturedImage( $text )
	{
		$pattern = '#<img class="featured[^>]*>#i';
		$pattern = '#<a class="easyblog-thumb-preview featured(.*)</a>#i';
		return preg_replace( $pattern , '' , $text , 1 );
	}

	public static function removeGallery( $text )
	{
		$pattern	= '#<div class="easyblog-placeholder-gallery"(.*)</div>#is';

		return preg_replace( $pattern , '' , $text );
	}

	public static function formatDraftBlog( $data )
	{
		$joomlaVersion	= EasyBlogHelper::getJoomlaVersion();
		$config     	= EasyBlogHelper::getConfig();

		if(! empty($data))
		{
			$modelPT	= self::getModel( 'PostTag' );

			for($i = 0; $i < count($data); $i++)
			{
				$row	= $data[$i];

				$profile = EasyBlogHelper::getTable( 'Profile', 'Table' );
				$profile->load( $row->created_by );
				$row->avatar		= $profile->getAvatar();
				$row->displayName	= $profile->getName();

				$date				= JFactory::getDate( $row->created );
				$row->totalComments	= 0;
				$row->isFeatured    = 0;

				$row->category      = (empty($row->category)) ? JText::_('COM_EASYBLOG_UNCATEGORIZED') : $row->category;
				$row->text			= '';

				$requireVerification = false;
				if($config->get('main_password_protect', true) && !empty($row->blogpassword))
				{
					$row->title	= JText::sprintf('COM_EASYBLOG_PASSWORD_PROTECTED_BLOG_TITLE', $row->title);
				}

				//blog summary
				$row->summary   = JString::substr(strip_tags($row->content), 0, 200);


				$row->introtext		= $row->intro;
				$row->excerpt		= $row->introtext;
				$row->content		= $row->text;

				//get number of tag count.
				$row->tagcount      = 0;

				$row->_tags			= array();
				$blogTags           = array();

				if( !empty($row->tags) )
				{
					$blogTags   = explode( ',', $row->tags);
				}

				if(! empty($blogTags))
				{
					$row->_tags		= $blogTags;
					$row->tagcount  = count( $blogTags );
				}
				else
				{
					$row->tags	= JText::_('COM_EASYBLOG_UNTAGGED');
				}

				$row->comments  = array();

			}//end foreach
		}//end if

		return $data;
	}

	public static function verifyBlogPassword($crypt, $id)
	{
		if(!empty($crypt) && !empty($id))
		{
			$jSession = JFactory::getSession();
			$password = $jSession->get('PROTECTEDBLOG_'.$id, '', 'EASYBLOG');

			if($crypt == $password)
			{
				return true;
			}
		}

		return false;
	}

	public static function processVideos( $content , $userId )
	{
		$config		= EasyBlogHelper::getConfig();
		$pattern	= '#"easyblog-video*[^>]*#is';
		$html		= $content;

		preg_match( $pattern , $content , $matches );

		if( !empty($matches) && isset( $matches[0 ] ) )
		{
			// Match the value
			$input		= $matches[ 0 ];
			preg_match( '/value="(.*})"/i' , $input , $data );

			if( isset( $data[1] ) )
			{
				require_once( EBLOG_CLASSES . DS . 'json.php' );

				$json		= new Services_JSON();
				$data[1]	= str_ireplace( '"' , '' , $data[ 1 ] );
				$obj		= $json->decode( $data[ 1 ] );

				$theme		= new CodeThemes();
				$uri		= rtrim( JURI::root() , '/' ) . '/' . str_ireplace( "\\" , '/' , rtrim( $config->get( 'main_image_path' ) , '/' ) ) . '/' . $userId;
				$uri		.= '/' . $obj->path;
				$storage	= str_ireplace( '/' , DS , rtrim( $config->get( 'main_image_path' ) , '/' ) );

				if( !isset( $obj->autoplay ) )
				{
					$obj->autoplay	= false;
				}

				$theme->set( 'uri'		, $uri );
				$theme->set( 'video'	, $obj );

				$html		= $theme->fetch( 'blog.video.php' );

				$html		= EasyBlogHelper::replaceVideo( $content , $html );
			}
		}

		return $html;
	}

	public static function replaceVideo( $content , $replacement )
	{
		$pattern	= array( '#<div class="easyblog-placeholder-video(.*?)</div>(.*?)</div>(.*?)<p>&nbsp;(.*?)</p>#is' );
		$html		= preg_replace( $pattern , $replacement , $content );

		// Remove unwanted data
		$html		= preg_replace( '#<p>&nbsp;</p>(|\n|\r\n)<p><input class="easyblog-video(.*?)</p>#is' , '' , $html );
		return $html;
	}

	public static function processGallery( $text , $userId )
	{
		// Lookup for div's with specific gallery classes.
		$pattern	= '#"easyblog-gallery[^>]*#is';
		$html		= '';

		EasyBlogHelper::removeGallery( $text );

		preg_match_all( $pattern , $text , $matches );

		if( $matches && count( $matches[0] ) > 0 )
		{
			$loaded	= array();

			foreach( $matches[0] as $match )
			{
				$input		= $match;

				preg_match( '/value="(.*\})"/i' , $input , $data );

				if( $data )
				{
					require_once( EBLOG_CLASSES . DS . 'json.php' );

					$json		= new Services_JSON();
					$obj		= $json->decode( $data[ 1 ] );

					if( !isset( $loaded[ $obj->path ] ) )
					{
						$config		= EasyBlogHelper::getConfig();
						$storage	= str_ireplace( '/' , DS , rtrim( $config->get( 'main_image_path' ) , '/' ) );
						$baseURI	= rtrim( JURI::root() , '/' ) . '/' . str_ireplace( '\\' , '/' , $storage ) . '/' . $userId . '/' . $obj->path;
						$basePath	= JPATH_ROOT . DS . $storage . DS . $userId . DS . $obj->path;

						// In case a folder is deleted, do not try to implement anything.
						if( !JFolder::exists( $basePath ) )
						{
							return $html;
						}

						$items		= JFolder::files( $basePath , '.' , false , false , array( 'index.html' ) );
						$images		= array();

						if( !$items )
						{
							return $html;
						}

						foreach( $items as $item )
						{
							// Test if thumbnail exists for this item.
							if( stristr( $item , EBLOG_MEDIA_THUMBNAIL_PREFIX ) === false )
							{
								// Test if there's thumbnails for it.
								$filePath	= $basePath . DS . $item;
								$thumbPath	= $basePath . DS . EBLOG_MEDIA_THUMBNAIL_PREFIX . $item;

								$image				= new stdClass();
								$image->original	= $item;

								if( JFile::exists( $thumbPath ) )
								{
									$image->thumbnail	= EBLOG_MEDIA_THUMBNAIL_PREFIX . $item;
								}
								else
								{
									$image->thumbnail	= $item;
								}

								$images[]	= $image;
							}
						}

						$theme		= new CodeThemes();
						$theme->set( 'gallery'	, $obj->name );
						$theme->set( 'baseURI'	, $baseURI );
						$theme->set( 'images'	, $images );

						$html		.= $theme->fetch( 'blog.gallery.php' );

						$loaded[ $obj->path ]	= true;
					}
				}
			}
		}

		return $html;
	}



	public static function formatBlog( $data , $loadComments = false , $removeFeaturedImage = true, $loadVideo = true , $frontpage = false )
	{
		JPluginHelper::importPlugin( 'easyblog' );
		$mainframe  = JFactory::getApplication();

		$params		= $mainframe->getParams('com_easyblog');
		$limitstart	= JRequest::getVar('limitstart', 0, '', 'int');
		$model		= EasyBlogHelper::getModel( 'Blog' );

		$joomlaVersion = EasyBlogHelper::getJoomlaVersion();

		$config     = EasyBlogHelper::getConfig();
		if(! empty($data))
		{
			$modelPT	= EasyBlogHelper::getModel( 'PostTag' );

			for($i = 0; $i < count($data); $i++)
			{
				$row	= $data[$i];

				$profile = EasyBlogHelper::getTable( 'Profile', 'Table' );
				$profile->load( $row->created_by );
				$row->avatar		= $profile->getAvatar();
				$row->avatarLink	= $profile->getProfileLink();
				$row->displayName	= $profile->getName();

				$date				= JFactory::getDate( $row->created );
				$idebateCode        = $config->get('comment_intensedebate_code');
				$disqusCode     	= $config->get('comment_disqus_code');
				$row->totalComments	= EasyBlogHelper::getHelper( 'Comment' )->getCommentCount( $row->id );
				$row->isFeatured    = EasyBlogHelper::isFeatured('post', $row->id);

				$row->category      = (empty($row->category)) ? JText::_('COM_EASYBLOG_UNCATEGORIZED') : JText::_( $row->category );
				$row->text			= '';
				$galleryHTML		= '';

				$requireVerification = false;
				if($config->get('main_password_protect', true) && !empty($row->blogpassword))
				{
					$row->title	= JText::sprintf('COM_EASYBLOG_PASSWORD_PROTECTED_BLOG_TITLE', $row->title);
					$requireVerification = true;
				}

				//here we process the video and get the links.
				if( $loadVideo )
				{
					$row->intro		= EasyBlogHelper::getHelper( 'Videos' )->processVideos( $row->intro );
					$row->excerpt   = EasyBlogHelper::getHelper( 'Videos' )->processVideos( $row->excerpt );
					$row->content	= EasyBlogHelper::getHelper( 'Videos' )->processVideos( $row->content );
					$row->text		= EasyBlogHelper::getHelper( 'Videos' )->processVideos( $row->text );
				}
				else
				{
					$row->intro		= EasyBlogHelper::getHelper( 'Videos' )->strip( $row->intro );
					$row->excerpt   = EasyBlogHelper::getHelper( 'Videos' )->strip( $row->excerpt );
					$row->content	= EasyBlogHelper::getHelper( 'Videos' )->strip( $row->content );
					$row->text		= EasyBlogHelper::getHelper( 'Videos' )->strip( $row->text );
				}

				$row->intro		= EasyBlogGoogleAdsense::stripAdsenseCode( $row->intro );
				$row->excerpt	= EasyBlogGoogleAdsense::stripAdsenseCode( $row->excerpt );
				$row->content	= EasyBlogGoogleAdsense::stripAdsenseCode( $row->content );
				$row->text		= EasyBlogGoogleAdsense::stripAdsenseCode( $row->text );


				if($requireVerification && !EasyBlogHelper::verifyBlogPassword($row->blogpassword, $row->id))
				{
					$theme = new CodeThemes();
					$theme->set('id', $row->id);
					$theme->set('return', base64_encode(EasyBlogRouter::_('index.php?option=com_easyblog&view=entry&id='.$row->id, false)));
					$row->text	= $theme->fetch( 'blog.protected.php' );
					$row->readmore	= false;
				}
				else
				{
					// @task: Process gallery items
					// Show gallery on frontpage if configured to do so.
					if( ( $config->get( 'main_image_gallery_frontpage') && $frontpage ) || !$frontpage )
					{
						$galleryHTML	= EasyBlogHelper::processGallery( $row->intro . $row->content , $row->created_by );
					}

					// @rule: Remove gallery text after it is processed.
					$row->intro		= EasyBlogHelper::removeGallery( $row->intro );
					$row->content	= EasyBlogHelper::removeGallery( $row->content );
					$teaser	= '';

					$maxCharacter   = $config->get('layout_maxlengthasintrotext', 150);

					// Decide whether or not to show read more link
					$row->readmore	= true;
					$truncate		= true;

					if( $config->get( 'layout_respect_readmore' ) )
					{
						// When introtext is not empty and content is empty
						if( !empty( $row->intro ) && empty($row->content) )
						{
							if( JString::strlen( strip_tags( $row->intro ) ) > $maxCharacter && $config->get( 'layout_blogasintrotext' ) )
							{
								$row->readmore	= true;
							}
							else
							{
								$row->readmore	= false;
							}
						}

						// Backward compatibility, this is probably from an older version
						if( empty( $row->intro ) && !empty($row->content) )
						{
							if( JString::strlen( strip_tags( $row->content ) ) > $maxCharacter )
							{
								$row->readmore	= true;
							}
							else
							{
								$row->readmore	= false;
							}
						}

						// New way of doing things where user explicitly set the read more line.
						if( !empty($row->intro) && !empty($row->content) )
						{
							$row->readmore	= true;
						}
					}


					// Check if truncation is really necessary because if introtext is already present, just use it.
					if( !empty($row->intro) && !empty($row->content) )
					{
						// We do not want the script to truncate anything since we'll just be using the intro part.
						$truncate			= false;
					}

					// If read more is not present and user wants to truncate the text
					if( $config->get('layout_blogasintrotext', 0) && $truncate )
					{
						$findImage      = false;

						// This is the combined content of the intro and the fulltext
						$content		= $row->intro . $row->content;

						// Paragraph type
						if( $config->get( 'main_truncate_type' ) == 'chars' )
						{
							$content		= EasyBlogHelper::removeZemantaTags( $content );

							// Remove video tags
							$content		= EasyBlogHelper::replaceVideo( $content  , '' );

							// Remove uneccessary html tags to avoid unclosed html tags
							$content		= strip_tags( $content );

							// Remove blank spaces since the word calculation should not include new lines or blanks.
							$content		= trim( $content );

							$maxCharacter   = ($maxCharacter <= 0) ? 150 : $maxCharacter;

							$row->text  	= JString::substr( $content , 0 , $maxCharacter) . '...';
						}
						elseif( $config->get( 'main_truncate_type' ) == 'words' )
						{
							$maxLength	= (int) $config->get( 'layout_maxlengthasintrotext', '50' );
							$tag		= false;
							$count		= 0;
							$output		= '';

							$content		= EasyBlogHelper::removeZemantaTags( $content );

							// Remove video tags
							$content		= EasyBlogHelper::replaceVideo( $content  , '' );

							// Remove uneccessary html tags to avoid unclosed html tags
							$content		= strip_tags( $content );

							$chunks		= preg_split("/([\s]+)/", $content, -1, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY);

							foreach($chunks as $piece)
							{

								if( !$tag || stripos($piece, '>') !== false )
								{
									$tag = (bool) (strripos($piece, '>') < strripos($piece, '<'));
								}

								if( !$tag && trim($piece) == '' )
								{
									$count++;
								}

								if( $count > $maxLength && !$tag )
								{
									break;
								}

								$output .= $piece;
							}

							unset($chunks);
							$row->text	= $output . '...';;
						}
						elseif( $config->get( 'main_truncate_type' ) == 'break')
						{
							$position	= 0;
							$matches	= array();
							$tag		= '<br />';

							$initialImage   = false;
							$afterImage     = false;

							$pattern		= '#<img[^>]*>#i';
							preg_match( $pattern , $row->intro . $row->content , $matches );
							if( $matches )
							{
								$initialImage   = true;
							}

							//reset
							unset($matches);
							$matches	= array();

							do
							{
								$position	= @JString::strpos( strtolower( $content ) , $tag , $position + 1 );

								if( $position !== false )
								{
									$matches[]	= $position;
								}
							} while( $position !== false );

							$maxTag		= (int) $config->get( 'main_truncate_maxtag' );

							if( count( $matches ) > $maxTag )
							{
								$row->text	= JString::substr( $content , 0 , $matches[ $maxTag - 1 ] + 6 ) . '...';
								$row->readmore	= true;
							}
							else
							{
								$row->text	= $content;
								$row->readmore	= false;
							}

							// test on the content after the truncation.
							//reset
							unset($matches);
							$matches	= array();

							preg_match( $pattern , $row->text , $matches );
							if( $matches )
							{
								$afterImage   = true;
							}

							// now we determine whether we should get the image teaser or not.
							if( $initialImage && !$afterImage )
							{
								$findImage  = true;
							}
						}
						else
						{
							$position	= 0;
							$matches	= array();
							$tag		= '</p>';

							$initialImage   = false;
							$afterImage     = false;

							$pattern		= '#<img[^>]*>#i';
							preg_match( $pattern , $content , $matches );
							if( $matches )
							{
								$initialImage   = true;
							}

							// reset variable $matches
							unset($matches);
							$matches = array();

							do
							{
								$position	= @JString::strpos( strtolower( $content ) , $tag , $position + 1 );

								if( $position !== false )
								{
									$matches[]	= $position;
								}
							} while( $position !== false );

							// @TODO: Configurable
							$maxTag		= (int) $config->get( 'main_truncate_maxtag' );

							if( count( $matches ) > $maxTag )
							{
								$row->text	= JString::substr( $content , 0 , $matches[ $maxTag - 1 ] + 4 ) . '...';

								// check if there is any leading <div> or not.
								$divTag 		= '<div';
								$divCloseTag 	= '</div>';
								$position   = 0;
								$divmatches	= array();

								do
								{
									$position	= strpos( strtolower( $row->text ) , $divTag , $position);
									if( $position !== false)
									{
										// now check if there is any closed div or not.
										$cposition = $position;
										$cposition = strpos( strtolower( $row->text ) , $divCloseTag , $cposition + 1);

										if( $cposition === false )
										{
											$divmatches[]	= $position;
											$position       = $cposition;
										}

										$position = $position + 1;
									}
								} while( $position !== false );

								if( count( $divmatches ) > 0 )
								{
									foreach( $divmatches as $div)
									{
										$row->text  = $row->text . '</div>';
									}
								}

								$row->readmore	= true;
							}
							else
							{
								$row->text	= $content;
								$row->readmore	= false;
							}

							//reset
							unset($matches);
							$matches = array();

							// test on the content after the truncation.
							preg_match( $pattern , $row->text , $matches );
							if( $matches )
							{
								$afterImage   = true;
							}

							// now we determine whether we should get the image teaser or not.
							if( $initialImage && !$afterImage )
							{
								$findImage  = true;
							}

						}

						if(  $config->get( 'main_truncate_type' ) == 'chars' ||  $config->get( 'main_truncate_type' ) == 'words' || $findImage )
						{
							// Append image in the post if truncation is done by characters
							if( $config->get( 'main_teaser_image' ) )
							{
								// Match images that has preview.
								$pattern		= '/<a class="easyblog-thumb-preview"(.*?)<\/a>/is';

								preg_match( $pattern , $row->intro . $row->content , $matches );

								// Legacy images that doesn't have previews.
								if( empty( $matches ) )
								{
									$pattern		= '#<img[^>]*>#i';

									preg_match( $pattern , $row->intro . $row->content , $matches );
								}

								if( !empty( $matches ) )
								{
									if( $config->get( 'main_teaser_image_align' ) == 'float-l' || $config->get( 'main_teaser_image_align') == 'float-r' )
									{
										$row->text	= '<div class="teaser-image clearfix ' . $config->get( 'main_teaser_image_align' ) . '" style="margin:8px;">' . $matches[ 0 ] . '</div>' . $row->text;
									}
									else
									{
										$row->text	.= '<div class="teaser-image clearfix" style="margin:8px;text-align: ' . $config->get( 'main_teaser_image_align' ) . ' !important;">' . $matches[ 0 ] . '</div>';
									}
								}
							}
						}
					}
					else
					{
						$row->text  	= empty( $row->intro ) ? $row->content : $row->intro;
					}


					// Append gallery html at the end of the item
					$row->text	.= $galleryHTML;

					// @task: Process video items
					$row->text		= EasyBlogHelper::processVideos( $row->text , $row->created_by );
				}

				// Remove first image from featured post
				if( $removeFeaturedImage && $row->isFeatured )
				{
					$row->text	= EasyBlogHelper::removeFeaturedImage( $row->text );
				}

				// @trigger: onEasyBlogPrepareContent
				EasyBlogHelper::triggerEvent( 'easyblog.prepareContent' , $row , $params , $limitstart );

				$row->introtext		= $row->intro;

				// @trigger: onPrepareContent / onContentPrepare
				EasyBlogHelper::triggerEvent( 'prepareContent' , $row , $params , $limitstart );

				$row->excerpt		= $row->introtext;
				$row->content		= $row->text;
				//onPrepareContent trigger end

				//get number of tag count.
				$row->tagcount      = 0;

				$blogTags			= $modelPT->getBlogTags($row->id);
				$arrBlogTags		= array();
				$row->_tags			= array();

				if(! empty($blogTags))
				{
					//build the tag link - TODO
					foreach($blogTags as $bt)
					{
						//$arrBlogTags[] = $bt->title;
						$arrBlogTags[] = '<a href="' . EasyBlogRouter::_('index.php?option=com_easyblog&view=tags&layout=tag&id=' . $bt->id ) . '">' . JText::_( $bt->title ) . '</a>';
					}
					$row->_tags		= $blogTags;
					$row->tagcount  = count( $arrBlogTags );
					$row->tags		= implode(', ', $arrBlogTags);
				}
				else
				{
					$row->tags	= JText::_('COM_EASYBLOG_UNTAGGED');
				}

				if( $loadComments )
				{
					$row->comments	= EasyBlogHelper::getHelper( 'Comment' )->getBlogComment( $row->id, $config->get( 'layout_showcommentcount' , 3 ) , 'desc', true );

					if(! empty( $row->comments ) )
					{
						for( $x = 0; $x < count( $row->comments ); $x++ )
						{
							$comment			= $row->comments[ $x ];

							$comment->comment	= EasyBlogCommentHelper::parseBBCode( $comment->comment );
						}
					}
				}
				else
				{
					$row->comments  = array();
				}

				$row->event = new stdClass();

				// @trigger: onContentAfterTitle / onAfterDisplayTitle
				$results		= EasyBlogHelper::triggerEvent( 'afterDisplayTitle' , $row , $params , $limitstart );
				$row->event->afterDisplayTitle	= JString::trim( implode( "\n" , $results ) );

				// @trigger: onContentAfterTitle / onAfterDisplayTitle
				$results		= EasyBlogHelper::triggerEvent( 'beforeDisplayContent' , $row , $params , $limitstart );
				$row->event->beforeDisplayContent	= JString::trim( implode( "\n" , $results ) );

				// @trigger: onContentAfterTitle / onAfterDisplayTitle
				$results		= EasyBlogHelper::triggerEvent( 'afterDisplayContent' , $row , $params , $limitstart );
				$row->event->afterDisplayContent	= JString::trim( implode( "\n" , $results ) );

				// Facebook Like integrations
				require_once( EBLOG_CLASSES . DS . 'facebook.php' );
				$facebookLike	= EasyBlogFacebookLikes::getLikeHTML( $row );
				$row->facebookLike = $facebookLike;

			}//end foreach
		}//end if

		return $data;
	}

	public static function removeZemantaTags( $content )
	{
		$output = preg_replace( '/<p class="zemanta-img-attribution"(.*?)<\/p>/is' , '' , $content );

		return $output;
	}

	public static function triggerEvent( $event , &$row , &$params , $limitstart )
	{
		$dispatcher = JDispatcher::getInstance();
		$version	= EasyBlogHelper::getJoomlaVersion();
		$events		= array(
								'1.5' => array(
													'easyblog.prepareContent'	=> 'onEasyBlogPrepareContent',
													'prepareContent' 			=> 'onPrepareContent',
													'afterDisplayTitle'			=> 'onAfterDisplayTitle',
													'beforeDisplayContent'		=> 'onBeforeDisplayContent',
													'afterDisplayContent'		=> 'onAfterDisplayContent',
													'beforeSave'				=> 'onBeforeContentSave'
												),
								'1.6' => array(
													'easyblog.prepareContent'	=> 'onEasyBlogPrepareContent',
													'prepareContent'			=> 'onContentPrepare',
													'afterDisplayTitle'			=> 'onContentAfterTitle',
													'beforeDisplayContent'		=> 'onContentBeforeDisplay',
													'afterDisplayContent'		=> 'onContentAfterDisplay',
													'beforeSave'				=> 'onContentBeforeSave'
												),
								'1.7' => array(
													'easyblog.prepareContent'	=> 'onEasyBlogPrepareContent',
													'prepareContent'			=> 'onContentPrepare',
													'afterDisplayTitle'			=> 'onContentAfterTitle',
													'beforeDisplayContent'		=> 'onContentBeforeDisplay',
													'afterDisplayContent'		=> 'onContentAfterDisplay',
													'beforeSave'				=> 'onContentBeforeSave'
												),
								'2.5' => array(
													'easyblog.prepareContent'	=> 'onEasyBlogPrepareContent',
													'prepareContent'			=> 'onContentPrepare',
													'afterDisplayTitle'			=> 'onContentAfterTitle',
													'beforeDisplayContent'		=> 'onContentBeforeDisplay',
													'afterDisplayContent'		=> 'onContentAfterDisplay',
													'beforeSave'				=> 'onContentBeforeSave'
												)
							);

		if( $version >= '1.6' )
		{
			return $dispatcher->trigger( $events[ $version ][ $event ] , array( 'easyblog.blog' , &$row , &$params , $limitstart ) );
		}
		return $dispatcher->trigger( $events[ $version ][ $event ] , array( &$row , &$params , $limitstart ) );
	}

	public static function getExternalLink($link, $xhtml = false)
	{
		$uri	= JURI::getInstance();
		$domain	= $uri->toString( array('scheme', 'host', 'port'));

		return $domain . '/' . ltrim(EasyBlogRouter::_( $link, $xhtml , null, true ), '/');
	}

	public static function makeFeatured($type, $contentId)
	{
		$db 	= JFactory::getDBO();
		$date   = JFactory::getDate();

		$query  = 'SELECT `id` FROM `#__easyblog_featured` WHERE `content_id` = ' . $db->Quote($contentId);
		$query  .= ' AND `type` = ' . $db->Quote($type);
		$db->setQuery($query);

		$fid    = $db->loadResult();

		if(empty($fid))
		{
			$obj    = new stdClass();

			$obj->content_id    = $contentId;
			$obj->type    		= $type;
			$obj->created    	= $date->toMySQL();

			$db->insertObject( '#__easyblog_featured' , $obj );

			if( $type == 'post' )
			{
				$blog		= EasyBlogHelper::getTable( 'Blog' , 'Table' );
				$blog->load( $contentId );

				$config				= EasyBlogHelper::getConfig();
				if( $config->get( 'integrations_jomsocial_feature_blog_activity' ) )
				{
					$lang		= JFactory::getLanguage();
					$lang->load( 'com_easyblog' , JPATH_ROOT );
					$title	= JString::substr( $blog->title , 0 , 30 ) . '...';

					$easyBlogItemid	= '';
					$mainframe 	= JFactory::getApplication();
					if( $mainframe->isAdmin() )
					{
						$easyBlogItemid	= EasyBlogRouter::getItemId('latest');
						$easyBlogItemid = '&Itemid=' . $easyBlogItemid;
					}

					$blogLink		= EasyBlogRouter::getRoutedURL( 'index.php?option=com_easyblog&view=entry&id='. $blog->id . $easyBlogItemid, false, true);
					$my				= JFactory::getUser();

					$blogContent    = $blog->intro . $blog->content;
					$blogContent	= EasyBlogHelper::getHelper( 'Videos' )->strip( $blogContent );
					$blogContent	= EasyBlogGoogleAdsense::stripAdsenseCode( $blogContent );

					$pattern		= '#<img[^>]*>#i';
					preg_match( $pattern , $blogContent , $matches );

					// Remove all html tags from the content as we want to chop it down.
					$blogContent	= strip_tags( $blogContent );

					if (strlen($blogContent) > $config->get( 'integrations_jomsocial_blogs_length', 250 ))
					{
						$blogContent = JString::substr($blogContent, 0, $config->get( 'integrations_jomsocial_blogs_length', 250 ) ) . ' ...';
					}


					if( $matches )
					{
						$matches[0]		= JString::str_ireplace( 'img ' , 'img style="margin: 0 5px 5px 0;float: left; height: auto; width: 120px !important;"' , $matches[0 ] );
						$blogContent	= $matches[0] . $blogContent . '<div style="clear: both;"></div>';
					}
					$blogContent	.= '<div style="text-align: right;"><a href="' . $blogLink . '">' . JText::_( 'COM_EASYBLOG_CONTINUE_READING' ) . '</a></div>';

					$title			= JText::sprintf( 'COM_EASYBLOG_JS_ACTIVITY_BLOG_FEATURED' , $blogLink , $title);
					EasyBlogHelper::addJomsocialActivity( $blog , 'easyblog.blog.featured' , $title , $my->id , $blogContent );
				}

				if( $config->get( 'integrations_mighty_activity_feature_blog' ) )
				{
					EasyBlogHelper::getHelper( 'MightyTouch' )->addFeaturedActivity( $blog );
				}
			}
		}

		return true;
	}

	public static function removeFeatured($type, $contentId)
	{
		$db = JFactory::getDBO();

		$query  = 'DELETE FROM `#__easyblog_featured` WHERE `content_id` = ' . $db->Quote($contentId);
		$query  .= ' AND `type` = ' . $db->Quote($type);

		$db->setQuery($query);
		$db->query();

		return true;
	}

	public static function isFeatured($type, $contentId)
	{
		static $blogs	= array();

		if( !isset( $blogs[ $type ][ $contentId ] ) )
		{
			$db = JFactory::getDBO();

			$query  = 'SELECT COUNT(1) FROM `#__easyblog_featured` WHERE `content_id` = ' . $db->Quote($contentId);
			$query  .= ' AND `type` = ' . $db->Quote($type);

			$db->setQuery($query);

			$result = $db->loadResult();
			$result = (empty($result)) ? 0 : $result;
			$blogs[ $type ][ $contentId ]	= $result;
		}
		return ( $blogs[ $type ][ $contentId ] > 0 );
	}

	public static function addJomSocialActivityBlog( $blog, $isNew, $isFeed = false )
	{
		$jsCoreFile	= JPATH_ROOT . DS . 'components' . DS . 'com_community' . DS . 'libraries' . DS . 'core.php';
		$config		= EasyBlogHelper::getConfig();
		$lang		= JFactory::getLanguage();
		$lang->load( 'com_easyblog' , JPATH_ROOT );

		// We do not want to add activities if new blog activity is disabled.
		if( $isFeed )
		{
			if( $isNew && !$config->get( 'integrations_jomsocial_rss_import_activity' ) )
			{
				return false;
			}
		}
		else
		{
			if( $isNew && !$config->get( 'integrations_jomsocial_blog_new_activity' ) )
			{
				return false;
			}
		}

		// We do not want to add activities if update blog activity is disabled.
		if( !$isNew && !$config->get( 'integrations_jomsocial_blog_update_activity') )
		{
			return false;
		}

		if( JFile::exists( $jsCoreFile ) )
		{
			require_once( $jsCoreFile );

			$blogCommand	= ($isNew) ? 'easyblog.blog.add' : 'easyblog.blog.update';

			$blogTitle		= htmlspecialchars( $blog->title );
			if (strlen($blogTitle) > 80)
			{
				$blogTitle		= JString::substr( $blog->title , 0 , 80 ) . '...';
			}

			$category		= EasyBlogHelper::getTable( 'ECategory' , 'Table' );
			$category->load( $blog->category_id );

			$easyBlogItemid	= '';
			$mainframe 	= JFactory::getApplication();
			if( $mainframe->isAdmin() )
			{
				$easyBlogItemid	= EasyBlogRouter::getItemId('latest');
				$easyBlogItemid = '&Itemid=' . $easyBlogItemid;
			}

			$blogLink		= EasyBlogRouter::getRoutedURL( 'index.php?option=com_easyblog&view=entry&id='. $blog->id . $easyBlogItemid, false, true);
			$categoryLink	= EasyBlogRouter::getRoutedURL( 'index.php?option=com_easyblog&view=categories&layout=listings&id=' . $category->id . $easyBlogItemid, false, true);

			$blogContent    = '';
			if($config->get('integrations_jomsocial_submit_content'))
			{

				$requireVerification = false;
				if($config->get('main_password_protect', true) && !empty($blog->blogpassword))
				{
					$row->title	= JText::sprintf('COM_EASYBLOG_PASSWORD_PROTECTED_BLOG_TITLE', $blog->title);
					$requireVerification = true;
				}

				$blogContent    = '';
				if($requireVerification && !EasyBlogHelper::verifyBlogPassword($blog->blogpassword, $blog->id))
				{
					$theme = new CodeThemes();
					$theme->set('id', $blog->id);
					$theme->set('return', base64_encode( $blogLink ) );
					$blogContent	= $theme->fetch( 'blog.protected.php' );
				}
				else
				{
					$blogContent    = $blog->intro . $blog->content;
					$blogContent	= EasyBlogHelper::getHelper( 'Videos' )->strip( $blogContent );
					$blogContent	= EasyBlogGoogleAdsense::stripAdsenseCode( $blogContent );

					$pattern		= '#<img[^>]*>#i';
					preg_match( $pattern , $blogContent , $matches );

					// Remove all html tags from the content as we want to chop it down.
					$blogContent	= strip_tags( $blogContent );
					$blogContent	= JString::substr($blogContent, 0, $config->get( 'integrations_jomsocial_blogs_length', 250 ) ) . ' ...';

					if( $matches )
					{
						$matches[0]		= JString::str_ireplace( 'img ' , 'img style="margin: 0 5px 5px 0;float: left; height: auto; width: 120px !important;"' , $matches[0 ] );
						$blogContent	= $matches[0] . $blogContent . '<div style="clear: both;"></div>';
					}
					$blogContent	.= '<div style="text-align: right;"><a href="' . $blogLink . '">' . JText::_( 'COM_EASYBLOG_CONTINUE_READING' ) . '</a></div>';
				}
			}

			$title  = ($isNew) ? JText::sprintf('COM_EASYBLOG_JS_ACTIVITY_BLOG_ADDED_NON_CATEGORY' , $blogLink , $blogTitle) : JText::sprintf('COM_EASYBLOG_JS_ACTIVITY_BLOG_UPDATED_NON_CATEGORY' , $blogLink , $blogTitle);
			if($config->get('integrations_jomsocial_show_category'))
			{
				$title  = ($isNew) ? JText::sprintf('COM_EASYBLOG_JS_ACTIVITY_BLOG_ADDED' , $blogLink , $blogTitle , $categoryLink , $category->title ) : JText::sprintf('COM_EASYBLOG_JS_ACTIVITY_BLOG_UPDATED' , $blogLink , $blogTitle , $categoryLink , $category->title );;
			}


			$obj			= new stdClass();
			$obj->access	= $blog->private;
			$obj->title		= $title;
			$obj->content	= $blogContent;
			$obj->cmd 		= $blogCommand;

			if( $config->get( 'integrations_jomsocial_activity_likes' ) )
			{
				$obj->like_id   = $blog->id;
				$obj->like_type = 'com_easyblog';
			}

			if( $config->get( 'integrations_jomsocial_activity_comments' ) )
			{
				$obj->comment_id    = $blog->id;
				$obj->comment_type  = 'com_easyblog';
			}

			$obj->actor   	= $blog->created_by;
			$obj->target  	= 0;
			$obj->app		= 'com_easyblog';
			$obj->cid		= $blog->id;

			// add JomSocial activities
			CFactory::load ( 'libraries', 'activities' );
			CActivityStream::add($obj);
		}
	}

	public static function addJomSocialActivityComment( $comment, $blogTitle )
	{
		$jsCoreFile	= JPATH_ROOT . DS . 'components' . DS . 'com_community' . DS . 'libraries' . DS . 'core.php';
		$config		= EasyBlogHelper::getConfig();

		// We do not want to add activities if new blog activity is disabled.
		if( !$config->get( 'integrations_jomsocial_comment_new_activity' ) )
		{
			return false;
		}

		if( JFile::exists( $jsCoreFile ) )
		{
			require_once( $jsCoreFile );

			$config		= EasyBlogHelper::getConfig();
			$command	= 'easyblog.comment.add';

			//$easyBlogItemid	= EasyBlogRouter::getItemId('latest');
			$blogTitle		= JString::substr( $blogTitle , 0 , 30 ) . '...';
			$blogLink		= EasyBlogRouter::getRoutedURL('index.php?option=com_easyblog&view=entry&id='. $comment->post_id, false, true);

			$content        = '';
			if($config->get('integrations_jomsocial_submit_content'))
			{
				$content		= $comment->comment;
				$content		= EasyBlogCommentHelper::parseBBCode( $content );
				$content		= nl2br( $content );
				$content		= strip_tags( $content );
				$content		= JString::substr( $content, 0 , $config->get( 'integrations_jomsocial_comments_length' ) );
			}

			$obj			= new stdClass();
			$obj->title		= JText::sprintf('COM_EASYBLOG_JS_ACTIVITY_COMMENT_ADDED' , $blogLink , $blogTitle );
			$obj->content	= ($config->get('integrations_jomsocial_submit_content')) ? $content : '';
			$obj->cmd 		= $command;
			$obj->actor   	= $comment->created_by;
			$obj->target  	= 0;
			$obj->app		= 'com_easyblog';
			$obj->cid		= $comment->id;

			if( $config->get( 'integrations_jomsocial_activity_likes' ) )
			{
				$obj->like_id   = $comment->id;
				$obj->like_type = 'com_easyblog.comments';
			}

			if( $config->get( 'integrations_jomsocial_activity_comments' ) )
			{
				$obj->comment_id    = $comment->id;
				$obj->comment_type  = 'com_easyblog.comments';
			}
			// add JomSocial activities
			CFactory::load ( 'libraries', 'activities' );
			CActivityStream::add($obj);
		}
	}

	public static function uploadAvatar( $profile, $isFromBackend = false )
	{
		jimport('joomla.utilities.error');
		jimport('joomla.filesystem.file');
		jimport('joomla.filesystem.folder');

		$my 		= JFactory::getUser();
		$mainframe	= JFactory::getApplication();
		$config		= EasyBlogHelper::getConfig();

		$acl		= EasyBlogACLHelper::getRuleSet();

		if(! $isFromBackend)
		{
			if(empty($acl->rules->upload_avatar))
			{
				$url  = 'index.php?option=com_easyblog&view=dashboard&layout=profile';
				EasyBlogHelper::setMessageQueue( JText::_('COM_EASYBLOG_NO_PERMISSION_TO_UPLOAD_AVATAR') , 'warning');
				$mainframe->redirect(EasyBlogRouter::_($url, false));
			}
		}
		$avatar_config_path = $config->get('main_avatarpath');
		$avatar_config_path = rtrim($avatar_config_path, '/');
		$avatar_config_path = str_replace('/', DS, $avatar_config_path);

		$upload_path		= JPATH_ROOT.DS.$avatar_config_path;
		$rel_upload_path	= $avatar_config_path;

		$err				= null;
		$file 				= JRequest::getVar( 'Filedata', '', 'files', 'array' );

		//check whether the upload folder exist or not. if not create it.
		if(! JFolder::exists($upload_path))
		{
			if(! JFolder::create( $upload_path ))
			{
				// Redirect
				if(! $isFromBackend)
				{
					EasyBlogHelper::setMessageQueue( JText::_('COM_EASYBLOG_IMAGE_UPLOADER_FAILED_TO_CREATE_UPLOAD_FOLDER') , 'error');
					self::setRedirect( EasyBlogRouter::_('index.php?option=com_easyblog&view=dashboard&layout=profile', false) );
				}
				else
				{
					//from backend
					self::setRedirect( EasyBlogRouter::_('index.php?option=com_easyblog&view=users', false), JText::_('COM_EASYBLOG_IMAGE_UPLOADER_FAILED_TO_CREATE_UPLOAD_FOLDER'), 'error' );
				}
				return;
			}
			else
			{
				// folder created. now copy index.html into this folder.
				if(! JFile::exists( $upload_path . DS . 'index.html' ) )
				{
					$targetFile = JPATH_ROOT . DS . 'components' . DS . 'com_easyblog' . DS . 'index.html';
					$destFile   = $upload_path . DS .'index.html';

					if( JFile::exists( $targetFile ) )
						JFile::copy( $targetFile, $destFile );
				}
			}
		}

		//makesafe on the file
		$file['name']	= $my->id . '_' . JFile::makeSafe($file['name']);

		if (isset($file['name']))
		{
			$target_file_path		= $upload_path;
			$relative_target_file	= $rel_upload_path.DS.$file['name'];
			$target_file 			= JPath::clean($target_file_path . DS. JFile::makeSafe($file['name']));
			$isNew                  = false;

			//include_once(JPATH_ROOT.DS.'components'.DS.'com_easyblog'.DS.'helpers'.DS.'image.php');
			require_once( EBLOG_HELPERS . DS . 'image.php' );
			require_once( EBLOG_CLASSES . DS . 'simpleimage.php' );

			if (! EasyImageHelper::canUpload( $file, $err ))
			{
				if(! $isFromBackend)
				{
					EasyBlogHelper::setMessageQueue( JText::_( $err ) , 'error');
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=dashboard&layout=profile', false));
				}
				else
				{
					//from backend
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=users', false), JText::_( $err ), 'error');
				}
				return;
			}

			if (0 != (int)$file['error'])
			{
				if(! $isFromBackend)
				{
					EasyBlogHelper::setMessageQueue( $file['error'] , 'error');
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=dashboard&layout=profile', false));
				}
				else
				{
					//from backend
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=users', false), $file['error'], 'error');
				}
				return;
			}

			//rename the file 1st.
			$oldAvatar 	= $profile->avatar;
			$tempAvatar	= '';

			if( $oldAvatar != 'default.png' && $oldAvatar != 'default_blogger.png' )
			{
				$session   = JFactory::getSession();
				$sessionId = $session->getToken();

				$fileExt 	= JFile::getExt(JPath::clean($target_file_path.DS.$oldAvatar));
				$tempAvatar	= JPath::clean($target_file_path . DS . $sessionId . '.' . $fileExt);

				JFile::move($target_file_path.DS.$oldAvatar, $tempAvatar);
			}
			else
			{
				$isNew  = true;
			}

			if (JFile::exists($target_file))
			{
				if( $oldAvatar != 'default.png' && $oldAvatar != 'default_blogger.png' )
				{
					//rename back to the previous one.
					JFile::move($tempAvatar, $target_file_path.DS.$oldAvatar);
				}

				if(! $isFromBackend)
				{
					EasyBlogHelper::setMessageQueue( JText::sprintf('ERROR.FILE_ALREADY_EXISTS', $relative_target_file) , 'error');
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=dashboard&layout=profile', false));
				}
				else
				{
					//from backend
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=users', false), JText::sprintf('ERROR.FILE_ALREADY_EXISTS', $relative_target_file), 'error');
				}
				return;
			}

			if (JFolder::exists($target_file))
			{

				if( $oldAvatar != 'default.png' && $oldAvatar != 'default_blogger.png' )
				{
					//rename back to the previous one.
					JFile::move($tempAvatar, $target_file_path.DS.$oldAvatar);
				}

				if(! $isFromBackend)
				{
					//JError::raiseNotice(100, JText::sprintf('ERROR.FOLDER_ALREADY_EXISTS',$relative_target_file));
					EasyBlogHelper::setMessageQueue( JText::sprintf('ERROR.FOLDER_ALREADY_EXISTS', $relative_target_file) , 'error');
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=dashboard&layout=profile', false));
				}
				else
				{
					//from backend
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=users', false), JText::sprintf('ERROR.FILE_ALREADY_EXISTS', $relative_target_file), 'error');
				}
				return;
			}

			$configImageWidth  = EBLOG_AVATAR_LARGE_WIDTH;
			$configImageHeight = EBLOG_AVATAR_LARGE_HEIGHT;

			$image = new SimpleImage();
			$image->load($file['tmp_name']);
			$image->resizeToFill($configImageWidth, $configImageHeight);
			$image->save($target_file, $image->image_type);

			//now we update the user avatar. If needed, we remove the old avatar.
			if( $oldAvatar != 'default.png' && $oldAvatar != 'default_blogger.png' )
			{
				//if(JFile::exists( JPATH_ROOT.DS.$oldAvatar ))
				if(JFile::exists( $tempAvatar ))
				{
					//JFile::delete( JPATH_ROOT.DS.$oldAvatar );
					JFile::delete( $tempAvatar );
				}
			}

			if($isNew && !$isFromBackend)
			{
				if( $my->id != 0 && $config->get('main_jomsocial_userpoint') )
				{
					$jsUserPoint	= JPATH_ROOT . DS . 'components' . DS . 'com_community' . DS . 'libraries' . DS . 'userpoints.php';
					if( JFile::exists( $jsUserPoint ) )
					{
						require_once( $jsUserPoint );
						CUserPoints::assignPoint( 'com_easyblog.avatar.upload' , $my->id );
					}
				}
			}

			return JFile::makeSafe( $file['name'] );
		}
		else
		{
			return 'default_blogger.png';
		}

	}


	public static function uploadCategoryAvatar( $category, $isFromBackend = false )
	{
		return EasyBlogHelper::uploadMediaAvatar( 'category', $category, $isFromBackend);
	}

	public static function uploadTeamAvatar( $team, $isFromBackend = false )
	{
		return EasyBlogHelper::uploadMediaAvatar( 'team', $team, $isFromBackend);
	}

	public static function uploadMediaAvatar( $mediaType, $mediaTable, $isFromBackend = false )
	{
		jimport('joomla.utilities.error');
		jimport('joomla.filesystem.file');
		jimport('joomla.filesystem.folder');

		$my 		= JFactory::getUser();
		$mainframe	= JFactory::getApplication();
		$config		= EasyBlogHelper::getConfig();

		$acl		= EasyBlogACLHelper::getRuleSet();


		// required params
		$layout_type    = ($mediaType == 'category') ? 'categories' : 'teamblogs';
		$view_type   	= ($mediaType == 'category') ? 'categories' : 'teamblogs';
		$default_avatar_type   	= ($mediaType == 'category') ? 'default_category.png' : 'default_team.png';



		if(! $isFromBackend && $mediaType == 'category')
		{
			if(empty($acl->rules->upload_cavatar))
			{
				$url  = 'index.php?option=com_easyblog&view=dashboard&layout='.$layout_type;
				EasyBlogHelper::setMessageQueue( JText::_('COM_EASYBLOG_NO_PERMISSION_TO_UPLOAD_AVATAR') , 'warning');
				$mainframe->redirect(EasyBlogRouter::_($url, false));
			}
		}

		$avatar_config_path = ($mediaType == 'category') ? $config->get('main_categoryavatarpath') : $config->get('main_teamavatarpath');
		$avatar_config_path = rtrim($avatar_config_path, '/');
		$avatar_config_path = str_replace('/', DS, $avatar_config_path);

		$upload_path		= JPATH_ROOT.DS.$avatar_config_path;
		$rel_upload_path	= $avatar_config_path;

		$err				= null;
		$file 				= JRequest::getVar( 'Filedata', '', 'files', 'array' );

		//check whether the upload folder exist or not. if not create it.
		if(! JFolder::exists($upload_path))
		{
			if(! JFolder::create( $upload_path ))
			{
				// Redirect
				if(! $isFromBackend)
				{
					EasyBlogHelper::setMessageQueue( JText::_('COM_EASYBLOG_IMAGE_UPLOADER_FAILED_TO_CREATE_UPLOAD_FOLDER') , 'error');
					self::setRedirect( EasyBlogRouter::_('index.php?option=com_easyblog&view=dashboard&layout='.$layout_type, false) );
				}
				else
				{
					//from backend
					self::setRedirect( EasyBlogRouter::_('index.php?option=com_easyblog&view='.$layout_type, false), JText::_('COM_EASYBLOG_IMAGE_UPLOADER_FAILED_TO_CREATE_UPLOAD_FOLDER'), 'error' );
				}
				return;
			}
			else
			{
				// folder created. now copy index.html into this folder.
				if(! JFile::exists( $upload_path . DS . 'index.html' ) )
				{
					$targetFile = JPATH_ROOT . DS . 'components' . DS . 'com_easyblog' . DS . 'index.html';
					$destFile   = $upload_path . DS .'index.html';

					if( JFile::exists( $targetFile ) )
						JFile::copy( $targetFile, $destFile );
				}
			}
		}

		//makesafe on the file
		$file['name']	= $mediaTable->id . '_' . JFile::makeSafe($file['name']);

		if (isset($file['name']))
		{
			$target_file_path		= $upload_path;
			$relative_target_file	= $rel_upload_path.DS.$file['name'];
			$target_file 			= JPath::clean($target_file_path . DS. JFile::makeSafe($file['name']));
			$isNew                  = false;

			//include_once(JPATH_ROOT.DS.'components'.DS.'com_easyblog'.DS.'helpers'.DS.'image.php');
			require_once( EBLOG_HELPERS . DS . 'image.php' );
			require_once( EBLOG_CLASSES . DS . 'simpleimage.php' );

			if (! EasyImageHelper::canUpload( $file, $err ))
			{
				if(! $isFromBackend)
				{
					EasyBlogHelper::setMessageQueue( JText::_( $err ) , 'error');
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=dashboard&layout='.$layout_type, false));
				}
				else
				{
					//from backend
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view='.$view_type, false), JText::_( $err ), 'error');
				}
				return;
			}

			if (0 != (int)$file['error'])
			{
				if(! $isFromBackend)
				{
					EasyBlogHelper::setMessageQueue( $file['error'] , 'error');
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=dashboard&layout='.$layout_type, false));
				}
				else
				{
					//from backend
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view='.$view_type, false), $file['error'], 'error');
				}
				return;
			}

			//rename the file 1st.
			$oldAvatar 	= (empty($mediaTable->avatar)) ? $default_avatar_type : $mediaTable->avatar;
			$tempAvatar	= '';
			if( $oldAvatar != $default_avatar_type)
			{
				$session   = JFactory::getSession();
				$sessionId = $session->getToken();

				$fileExt 	= JFile::getExt(JPath::clean($target_file_path.DS.$oldAvatar));
				$tempAvatar	= JPath::clean($target_file_path . DS . $sessionId . '.' . $fileExt);

				JFile::move($target_file_path.DS.$oldAvatar, $tempAvatar);
			}
			else
			{
				$isNew  = true;
			}

			if (JFile::exists($target_file))
			{
				if( $oldAvatar != $default_avatar_type)
				{
					//rename back to the previous one.
					JFile::move($tempAvatar, $target_file_path.DS.$oldAvatar);
				}

				if(! $isFromBackend)
				{
					EasyBlogHelper::setMessageQueue( JText::sprintf('ERROR.FILE_ALREADY_EXISTS', $relative_target_file) , 'error');
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=dashboard&layout='.$layout_type, false));
				}
				else
				{
					//from backend
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view='.$view_type, false), JText::sprintf('ERROR.FILE_ALREADY_EXISTS', $relative_target_file), 'error');
				}
				return;
			}

			if (JFolder::exists($target_file))
			{

				if( $oldAvatar != $default_avatar_type)
				{
					//rename back to the previous one.
					JFile::move($tempAvatar, $target_file_path.DS.$oldAvatar);
				}

				if(! $isFromBackend)
				{
					//JError::raiseNotice(100, JText::sprintf('ERROR.FOLDER_ALREADY_EXISTS',$relative_target_file));
					EasyBlogHelper::setMessageQueue( JText::sprintf('ERROR.FOLDER_ALREADY_EXISTS', $relative_target_file) , 'error');
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view=dashboard&layout='.$layout_type, false));
				}
				else
				{
					//from backend
					$mainframe->redirect(EasyBlogRouter::_('index.php?option=com_easyblog&view='.$view_type, false), JText::sprintf('ERROR.FILE_ALREADY_EXISTS', $relative_target_file), 'error');
				}
				return;
			}

			$configImageWidth  = EBLOG_AVATAR_LARGE_WIDTH;
			$configImageHeight = EBLOG_AVATAR_LARGE_HEIGHT;

			$image = new SimpleImage();
			$image->load($file['tmp_name']);
			$image->resizeToFill($configImageWidth, $configImageHeight);
			$image->save($target_file, $image->image_type);

			//now we update the user avatar. If needed, we remove the old avatar.
			if( $oldAvatar != $default_avatar_type)
			{
				if(JFile::exists( $tempAvatar ))
				{
					JFile::delete( $tempAvatar );
				}
			}

			return JFile::makeSafe( $file['name'] );
		}
		else
		{
			return $default_avatar_type;
		}

	}

	public static function unPublishPost()
	{
		$db 		= JFactory::getDBO();
		$date   	= JFactory::getDate();

		$query  = 'UPDATE `#__easyblog_post`';
		$query  .= ' SET `published` = ' . $db->Quote('0');
		$query	.= ' WHERE `publish_down` > `publish_up`';
		$query	.= ' AND `publish_down` <= ' . $db->Quote($date->toMySQL());
		$query	.= ' AND `publish_down` != ' . $db->Quote('0000-00-00 00:00:00');
		$query  .= ' AND `published` != ' . $db->Quote('0');
		$query  .= ' AND `published` != ' . $db->Quote('3');
		$query  .= ' AND `ispending` = ' . $db->Quote('0');


		$db->setQuery($query);
		$db->query();
	}


	public static function processScheduledPost( $max = 5 )
	{
		JTable::addIncludePath( JPATH_ROOT . DS . 'administrator' . DS . 'components' . DS . 'com_easyblog' . DS . 'tables' );

		$db 		= JFactory::getDBO();
		$date   	= JFactory::getDate();
		$config 	= EasyBlogHelper::getConfig();

		$query  = 'SELECT * FROM `#__easyblog_post`';
		$query	.= ' WHERE `publish_up` <= ' . $db->Quote($date->toMySQL());
		$query  .= ' AND `published` = ' . $db->Quote('2');
		$query  .= ' AND `ispending` = ' . $db->Quote('0');
		$query  .= ' ORDER BY `id`';

		if($max)
		{
			$query  .= ' LIMIT ' . $max;
		}

		$db->setQuery($query);
		$blogs  	= $db->loadObjectList();

		if( !$blogs )
		{
			return;
		}
		require_once( EBLOG_HELPERS . DS . 'subscription.php' );

		foreach($blogs as $item)
		{
			$blog = EasyBlogHelper::getTable( 'Blog', 'Table' );
			$blog->bind( $item );

			$query  = 'update `#__easyblog_post` set published = ' . $db->Quote( '1' );
			$query .= ' where id = ' . $db->Quote($blog->id);
			$db->setQuery($query);
			$db->query();

			// @Task: Add jomsocial activities
			EasyBlogHelper::addJomSocialActivityBlog($blog, $blog->isnew);

			if($blog->isnew)
			{
				//send notification to admin.
				JTable::addIncludePath( EBLOG_TABLES );
				$user		= JFactory::getUser($blog->created_by);
				$blogger 	= EasyBlogHelper::getTable( 'Profile', 'Table' );
				$blogger->setUser($user);

				// the sequece of the email data will be the same as the sequence in jtext::sprint params
				$emailData['blogTitle']		= $blog->title;
				$emailData['blogAuthor']	= $blogger->getName();
				$emailData['blogAuthorEmail']	= $user->email;
				$emailData['blogIntro']			= $blog->intro;

				$emailData['blogLink']		= EasyBlogRouter::getRoutedURL( 'index.php?option=com_easyblog&view=entry&id='. $blog->id, false, true);
				$emailData['blogContent']	= strip_tags($blog->intro . $blog->content, '<p><img><a><div><span><table><tr><td>');

				$notify	= EasyBlogHelper::getNotification();

				// send notification to admins
				if($config->get('notification_blogadmin'))
				{
					if($config->get('notification_email', '') != '')
					{
						$toNotifyEmails   = explode(',', $config->get('notification_email'));
						$notify->sendEmails($toNotifyEmails, JText::sprintf('COM_EASYBLOG_EMAIL_TITLE_NEW_BLOG_ADDED_WITH_TITLE' , JString::substr( $blog->title , 0 , 25 ) ) , '', 'email.blog.new.php', $emailData);
					}
					else
					{
						$notify->send('', 'admin', JText::sprintf('COM_EASYBLOG_EMAIL_TITLE_NEW_BLOG_ADDED_WITH_TITLE' , JString::substr( $blog->title , 0 , 25 ) ) , '', 'email.blog.new.php', $emailData);
					}
				}

				if( $config->get( 'notification_allmembers' ) )
				{
					EasyBlogSubscription::blogSendToAll( $blog, $emailData );
				}
				else
				{
					// process all subscroptions on new blog post.
					$blogContribution	= $blog->getTeamContributed();
					EasyBlogSubscription::processNewBlogNotification( $blog, $emailData, $blogContribution );
				}


				// ping to ping-o-matic
				if($config->get('main_pingomatic', 0))
				{
					//if pingomatic enabled...send the pings.
					require_once( EBLOG_CLASSES . DS . 'pingomatic.php' );

					$siteURL    = EasyBlogHelper::getExternalLink('index.php?option=com_easyblog&view=entry&id=' . $blog->id, true);
					$siteTitle  = htmlspecialchars($blog->title);

					$pingo  	= new EasyBlogPingomatic();
					$pingoResp  = $pingo->ping($siteTitle, $siteURL);
				}

			}

			$curDate            = JFactory::getDate();

			$blog->published    = 1;
			$blog->isnew    	= '0';
			$blog->modified     = $curDate->toMySQL();
			$blog->store();





			if( $blog->published == POST_ID_PUBLISHED )
			{
				$allowed	= array( EBLOG_OAUTH_LINKEDIN , EBLOG_OAUTH_FACEBOOK , EBLOG_OAUTH_TWITTER );

				// @rule: Process centralized options first
				// See if there are any global postings enabled.
				foreach( $allowed as $item )
				{
					if( $config->get( 'integrations_' . $item . '_centralized' ) && $config->get( 'integrations_' . $item . '_centralized_auto_post' ) )
					{
						EasyBlogSocialShareHelper::share( $blog , constant( 'EBLOG_OAUTH_' . JString::strtoupper( $item ) ) , true );
					}
				}

				// @rule: Process blogger autoposting options
				foreach( $allowed as $item )
				{
					if( in_array( $item , $allowed ) && $config->get( 'integrations_' . $item ) )
					{
						EasyBlogSocialShareHelper::share( $blog , constant( 'EBLOG_OAUTH_' . JString::strtoupper( $item ) ) );
					}
				}
			}
		}
	}

	/**
	 * Allows caller to detect specific css files from site's template
	 * and load it into the headers if necessary.
	 *
	 * @param	string $fileName
	 */
	public static function addTemplateCss( $fileName )
	{
		$document		= JFactory::getDocument();
		$document->addStyleSheet( rtrim(JURI::root(), '/') . '/components/com_easyblog/assets/css/' . $fileName );

		$mainframe		= JFactory::getApplication();
		$templatePath	= JPATH_ROOT . DS . 'templates' . DS . $mainframe->getTemplate() . DS . 'html' . DS . 'com_easyblog' . DS . 'assets' . DS . 'css' . DS . $fileName;

		if( JFile::exists($templatePath) )
		{
			$document->addStyleSheet( rtrim(JURI::root(), '/') . '/templates/' . $mainframe->getTemplate() . '/html/com_easyblog/assets/css/' . $fileName );

			return true;
		}

		return false;
	}

	/*
	 * Loads necessary dependency for the module stylings.
	 *
	 * @param	null
	 * @return	null
	 */
	public static function loadModuleCss()
	{
		static $loaded	= false;

		if( !$loaded )
		{
			$document	= JFactory::getDocument();

			EasyBlogHelper::addTemplateCss( 'module.css' );

			$document->addStyleSheet( rtrim(JURI::root(), '/') . '/components/com_easyblog/assets/css/fancybox/fancybox.css' );



			$loaded		= true;
		}
		return $loaded;
	}

	/**
	 * Method to load the theme CSS file based on the config
	 *
	 * @param	String		Filename to be loaded
	 * @param	Boolean		Is this a dashboard?
	 */
	public static function loadThemeCss( $file, $is_dashboard = false )
	{
		$document 	= JFactory::getDocument();
		$mainframe	= JFactory::getApplication();
		$config 	= EasyBlogHelper::getConfig();

		$template 	= $mainframe->getTemplate();

		$bloggertheme			= EasyBlogHelper::getBloggerTheme();
		$enableBloggerTheme 	= $config->get('layout_enablebloggertheme', true);
		$availableBloggerTheme 	= $config->get('layout_availablebloggertheme');

		if( $config->get( 'main_media_lightbox_preview' ) )
		{
			$url	= rtrim( JURI::root() , '/' ) . '/components/com_easyblog/assets/css/fancybox/fancybox.css';
			$document->addStyleSheet( $url );
		}

		if( !is_array( $availableBloggerTheme ) )
		{
			$availableBloggerTheme	= explode( '|' , $config->get('layout_availablebloggertheme') );
		}

		if($enableBloggerTheme && !empty($bloggertheme) && $bloggertheme != 'global' && in_array($bloggertheme, $availableBloggerTheme))
		{
			$theme = $bloggertheme;
			JRequest::setVar('theme', $bloggertheme);
		}
		else
		{
			$theme = $config->get( 'layout_theme' );
		}

		// @rule: Use Google fonts if necessary.
		$headingFont	= $config->get( 'layout_googlefont' );
		if(  $headingFont != 'site' )
		{
			// Replace the word Bold with :Bold
			$headingFont	= JString::str_ireplace( ' bold' , ':bold' , $headingFont );

			// Replace spaces with +
			$headingFont	= JString::str_ireplace( ' ' , '+' , $headingFont );

			$url			= 'https://fonts.googleapis.com/css?family=' . $headingFont;
			$document->addStyleSheet( $url );
		}

		$googlePlus		= $config->get( 'main_googleone', 0 );
		$socialFrontEnd	= $config->get( 'main_googleone_frontpage', 0 );

		if( $socialFrontEnd && $googlePlus)
		{
			$googlPlusUrl   = 'https://apis.google.com/js/plusone.js';
			$document->addScript( $googlPlusUrl );
		}

		$path		= null;
		$direction	= $document->direction;

		 /**
		 * for theme development purpose
		 */
		$usethis	= JRequest::getWord('theme');

		if ( !empty( $usethis ) ) {
			JRequest::setVar('theme', $usethis);
			$theme	= $usethis;
		}


		// new in 1.1
		// load the file based on the theme's config.ini
		$themeConfig = EasyBlogHelper::getThemeInfo( $theme );

		/**
		 * Load blog theme file
		 *
		 * Priority level
		 *
		 * 1. /templates/<joomla_template>/html/com_easyblog/
		 * 2. /components/com_easyblog/themes/<selected_theme>/
		 * 3. /components/com_easyblog/themes/<parent_for_selected_theme>/
		 * 4. /components/com_easyblog/themes/default/
		 */
		$override	= false;
		if ( JFile::exists( JPATH_ROOT . DS . 'templates' . DS . $template . DS . 'html' . DS . 'com_easyblog' . DS . 'css' . DS . $file ) )
		{
			$path		= rtrim( JURI::root(), '/' ) . '/templates/' . $template . '/html/com_easyblog/css/' . $file;
			$override	= true;
		}
		elseif( JFile::exists( EBLOG_THEMES . DS . $theme . DS . 'css' . DS . $file ) )
		{
			if ( $themeConfig->get('parent_css') )
			{
				$path	= rtrim( JURI::root(), '/' ) . '/components/com_easyblog/themes/' . $themeConfig->get('parent') . '/css/' . $file;
				$document->addStylesheet( $path );
			}
			$path	= rtrim( JURI::root(), '/' ) . '/components/com_easyblog/themes/' . $theme . '/css/' . $file;
			$override	= $theme != 'default' ? true : false;
		}

		else
		{
			$path	= rtrim( JURI::root(), '/' ) . '/components/com_easyblog/themes/default/css/' . $file;
		}

		// Always load the system css if template override in use as we want to allow them to
		// reuse the styling as much as possible.
		if( $override )
		{
			$document->addStylesheet( rtrim( JURI::root() , '/' ) . '/components/com_easyblog/themes/default/css/styles.css' );
		}
		$document->addStylesheet( $path );


		/**
		 * Load dashboard theme file
		 *
		 * Priority level
		 *
		 * 1. /templates/<joomla_template>/html/com_easyblog/dashboard/
		 * 2. /components/com_easyblog/themes/<selected_theme>/dashboard/
		 * 3. /components/com_easyblog/themes/dashboard/
		 */
		if ( $is_dashboard )
		{
			$dashboardTheme	= JString::trim( JString::strtolower( $config->get( 'layout_dashboard_theme' ) ) );
			$overridePath	= JPATH_ROOT . DS . 'templates' . DS . $template . DS . 'html' . DS . 'com_easyblog' . DS . 'dashboard' . DS . 'css' . DS . $file;
			$themePath		= JFile::exists( EBLOG_THEMES . DS . 'dashboard' . DS . $dashboardTheme . DS . 'css' . DS . $file );

			// Always load the base css file.
			$document->addStyleSheet( rtrim( JURI::root(), '/' ) . '/components/com_easyblog/themes/dashboard/system/css/styles.css' );

			if ( JFile::exists( $overridePath ) )
			{
				$path = rtrim( JURI::root(), '/' ) . '/templates/' . $template . '/html/com_easyblog/dashboard/css/' . $file;
			}
			elseif( $themePath )
			{
				$path	= rtrim( JURI::root(), '/' ) . '/components/com_easyblog/themes/dashboard/' . $dashboardTheme . '/css/' . $file;
			}
			else
			{
				$path	= rtrim( JURI::root(), '/' ) . '/components/com_easyblog/themes/dashboard/system/' . $file;
			}
			$document->addStylesheet( $path );
		}

		/**
		 * Load RTL file for default theme
		 */
		if ( $direction == 'rtl' )
		{
			// we need default RTL be loaded always
			$path	= rtrim( JURI::root(), '/' ) . '/components/com_easyblog/themes/default/css/rtl.css';
			$document->addStylesheet( $path );


			// checking the template override folder
			if ( JFile::exists( JPATH_ROOT . DS . 'templates' . DS . $template . DS . 'html' . DS . 'com_easyblog' . DS . 'css' . DS . 'rtl.css' ) )
			{
				$path = rtrim( JURI::root(), '/' ) . '/templates/' . $template . '/html/com_easyblog/css/rtl.css';
				$document->addStylesheet( $path );
			}


			// check if we need to load the parent CSS based on theme's config.ini
			// since 1.1
			if ( $themeConfig->get('parent_css') )
			{
				// now check if the parent theme have th RTL file
				if ( JFile::exists( EBLOG_THEMES . DS . $themeConfig->get('parent') . DS . 'css' . DS . 'rtl.css' ) )
				{
					$path	= rtrim( JURI::root(), '/' ) . '/components/com_easyblog/themes/' . $themeConfig->get('parent') . '/css/rtl.css';
					$document->addStylesheet( $path );
				}
			}


			// checking the theme folder
			if( JFile::exists( EBLOG_THEMES . DS . $theme . DS . 'css' . DS . 'rtl.css' ) )
			{
				$path	= rtrim( JURI::root(), '/' ) . '/components/com_easyblog/themes/' . $theme . '/css/rtl.css';
				$document->addStylesheet( $path );
			}
		}


		/**
		 * Load RTL file for dashboard theme
		 */
		if ( $direction == 'rtl' && $is_dashboard )
		{
			if ( JFile::exists( JPATH_ROOT . DS . 'templates' . DS . $template . DS . 'html' . DS . 'com_easyblog' . DS . 'dashboard' . DS . 'css' . DS . 'rtl.css' ) )
			{
				$path = rtrim( JURI::root(), '/' ) . '/templates/' . $template . '/html/com_easyblog/dashboard/css/rtl.css';
			}
			elseif( JFile::exists( EBLOG_THEMES . DS . 'themes' . DS . 'dashboard' . DS . $template . DS . 'css' . DS . 'rtl.css' ) )
			{
				$path	= rtrim( JURI::root(), '/' ) . '/components/com_easyblog/themes/dashboard/' . $template . '/css/rtl.css';
			}
			else
			{
				$path	= rtrim( JURI::root(), '/' ) . '/components/com_easyblog/themes/dashboard/system/css/rtl.css';
			}
			$document->addStylesheet( $path );
		}
	}

	function getUserId( $username )
	{
		$db				= JFactory::getDBO();

		$query  = 'SELECT `id` FROM #__easyblog_users WHERE `permalink`=' . $db->Quote( $username );
		$db->setQuery( $query );
		$result = $db->loadResult();

		if(empty($result))
		{
			$query	= 'SELECT `id` FROM #__users WHERE `username`=' . $db->Quote( $username );
			$db->setQuery( $query );
			$result = $db->loadResult();
		}

		return $result;
	}

	public static function getBlogNavigation( $blogId, $creationDate, $typeId = '0', $type = 'sitewide')
	{
		$db 	= JFactory::getDBO();
		$my 	= JFactory::getUser();
		$config = EasyBlogHelper::getConfig();

		$keys   = array('prev','next');
		$nav    = array();
		$nav['prev']    = null;
		$nav['next']    = null;

		$isBloggerMode  = EasyBlogRouter::isBloggerMode();


		$menus	= JFactory::getApplication()->getMenu();
		$menu	= $menus->getActive();
		$queryInclude 	= '';

		if( is_object( $menu ) )
		{
			$params 	= new JParameter( $menu->params );
			$cats 		= $params->get( 'inclusion' );

			if( !is_array( $cats ) )
			{
				$cats 	= array( $cats );
			}

			$queryInclude	= ' AND a.`category_id` IN (';

			foreach( $cats as $allowedCat )
			{
				$queryInclude .= $db->Quote( $allowedCat );

				if( next( $cats ) !== false )
				{
					$queryInclude .= ',';
				}
			}
			$queryInclude 	.= ')';
		}

		// get all private categories id
		$excludeCats	= EasyBlogHelper::getPrivateCategories();
		$queryExclude   = '';
		if(! empty($excludeCats))
		{
			$queryExclude .= ' AND a.`category_id` NOT IN (';

			for( $i = 0; $i < count( $excludeCats ); $i++ )
			{
				$queryExclude	.= $db->Quote( $excludeCats[ $i ] );

				if( next( $excludeCats ) !== false )
				{
					$queryExclude .= ',';
				}
			}
			$queryExclude	.= ')';
		}

		foreach($keys as $key)
		{

			$query  = 'SELECT a.`id`, a.`title`';
			$query  .= ' FROM `#__easyblog_post` AS `a`';

			if($type == 'team' && ! empty($typeId))
			{
				$query  .= ' INNER JOIN `#__easyblog_team_post` AS `b`';
				$query  .= ' 	ON a.`id` = b.`post_id`';
			}

			$query  .= ' WHERE a.`published` = ' . $db->Quote('1');
			$query  .= ' AND a.`ispending` = ' . $db->Quote('0');


			//blog privacy setting
			// @integrations: jomsocial privacy
			$file		= JPATH_ROOT . DS . 'components' . DS . 'com_community' . DS . 'libraries' . DS . 'core.php';
			if( $config->get( 'main_jomsocial_privacy' ) && JFile::exists( $file ) && !EasyBlogHelper::isSiteAdmin() )
			{
				require_once( $file );

				$my			= JFactory::getUser();
				$jsFriends	= CFactory::getModel( 'Friends' );
				$friends	= $jsFriends->getFriendIds( $my->id );

				// Insert query here.
				$query	.= ' AND (';
				$query	.= ' (a.`private`= 0 ) OR';
				$query	.= ' ( (a.`private` = 20) AND (' . $db->Quote( $my->id ) . ' > 0 ) ) OR';

				if( empty( $friends ) )
				{
					$query	.= ' ( (a.`private` = 30) AND ( 1 = 2 ) ) OR';
				}
				else
				{
					$query	.= ' ( (a.`private` = 30) AND ( a.' . $db->nameQuote( 'created_by' ) . ' IN (' . implode( ',' , $friends ) . ') ) ) OR';
				}

				$query	.= ' ( (a.`private` = 40) AND ( a.' . $db->nameQuote( 'created_by' ) .'=' . $my->id . ') )';
				$query	.= ' )';
			}
			else
			{
				//blog privacy setting
				if($my->id == 0)
					$query .= ' AND a.`private` = ' . $db->Quote(BLOG_PRIVACY_PUBLIC);
			}

			//include categories
			if( !empty( $queryInclude ) )
			{
				$query	.= $queryInclude;
			}

			//exclude private categories
			$query	.= $queryExclude;


			if($isBloggerMode !== false)
				$query .= ' AND a.`created_by` = ' . $db->Quote($isBloggerMode);


			if($type == 'team' && ! empty($typeId))
			{
				$query  .= ' 	AND b.`team_id` = ' . $db->Quote($typeId);
			}
			else
			{
				$query  .= ' 	AND a.`issitewide` = ' . $db->Quote('1');
			}

			if($key == 'prev')
			{
				$query	.= ' AND a.`created` < ' . $db->Quote($creationDate);
				$query  .= ' ORDER BY a.`created` DESC';
			}
			else if($key == 'next')
			{
				$query	.= ' AND a.`created` > ' . $db->Quote($creationDate);
				$query  .= ' ORDER BY a.`created` ASC';
			}

			$query  .= ' LIMIT 1';

			$db->setQuery($query);
			$result = $db->loadObjectList();

			$nav[$key]  = $result;
		}

		return $nav;
	}


	public static function setMeta( $id, $type, $defaultViewDesc = '')
	{
		$config = EasyBlogHelper::getConfig();

		$db = JFactory::getDBO();

		$query = 'SELECT id, keywords, description FROM ' . $db->nameQuote('#__easyblog_meta') . ' WHERE content_id = ' . $db->Quote($id) . ' and type = ' . $db->Quote($type);

		$db->setQuery($query);

		$result = $db->loadObject();

		//auto fill meta keywords if the option is enable and the user did not set his own meta keywords.
		if($type==META_TYPE_POST && ( ($config->get('main_meta_autofillkeywords') && empty($result->keywords) )|| ($config->get( 'main_meta_autofilldescription') )) )
		{
			$keywords = array();

			//set category into the meta keyword.
			$post	= EasyBlogHelper::getTable( 'Blog' , 'Table' );
			$post->load( $id );
			$category	= EasyBlogHelper::getTable( 'ECategory' , 'Table' );
			$category->load( $post->category_id );
			$keywords[] = $category->title;

			if( $config->get( 'main_meta_autofillkeywords') && empty($result->keywords) )
			{
				//set tags into the meta keyword.
				$modelPT	= self::getModel( 'PostTag' );
				$blogTags	= $modelPT->getBlogTags($id);

				if(!empty($blogTags))
				{
					foreach($blogTags as $tag)
					{
						$keywords[] = $tag->title;
					}
				}
				$result = new stdClass();
				$result->keywords = implode(',', $keywords);
			}


			if( $config->get( 'main_meta_autofilldescription' ) && empty($result->description) )
			{
				$content				= !empty( $post->intro ) ? strip_tags( $post->intro ) : strip_tags( $post->content );
				$content				= str_ireplace( "\r\n" , "" , $content );

				// @rule: Set description into meta headers
				$result->description	= JString::substr( $content , 0 , $config->get( 'main_meta_autofilldescription_length' ) );
			}

		}

		if ( !empty( $result ) ) {
			$document = JFactory::getDocument();

			if ( !empty( $result->keywords ) )
			{
				$document->setMetadata('keywords', $result->keywords);
			}

			if ( !empty( $result->description ) )
			{
				$document->setMetadata('description', $result->description);
			}
			else
			{
				if( !empty( $defaultViewDesc) )
				{
					//get joomla default description.
					$jConfig    = JFactory::getConfig();
					$joomlaDesc = $jConfig->getValue('MetaDesc');

					$metaDesc   = $defaultViewDesc . ' - ' . $joomlaDesc;
					$document->setMetadata('description', $metaDesc);
				}
			}
		}
	}

	public static function addLikes($contentId, $type, $userId = '0')
	{
		if($userId == '0')
		{
			$user   = JFactory::getUser();
			$userId = $user->id;
		}

		$date   = JFactory::getDate();
		$likes 	= EasyBlogHelper::getTable( 'Likes', 'Table' );

		$params   = array();
		$params['type'] 		= $type;
		$params['content_id'] 	= $contentId;
		$params['created_by'] 	= $userId;
		$params['type'] 		= $type;
		$params['created'] 		= $date->toMySQL();

		$likes->bind($params);
		$likes->store();
		return $likes->id;
	}

	public static function removeLikes($likesId)
	{
		$likes 	= EasyBlogHelper::getTable( 'Likes', 'Table' );
		$likes->load($likesId);

		$likes->delete();
		return true;
	}

	public static function getLikesAuthors($contentId, $type, $userId)
	{
		$db     = JFactory::getDBO();
		$config = EasyBlogHelper::getConfig();

		$displayFormat  = $config->get('layout_nameformat');
		$displayName    = '';

		switch($displayFormat){
			case "name" :
				$displayName = 'a.name';
				break;
			case "username" :
				$displayName = 'a.username';
				break;
			case "nickname" :
			default :
				$displayName = 'b.nickname';
				break;
		}

		$query	= 'select a.id as `user_id`, c.id, ' . $displayName . ' as `displayname`';
		$query	.= ' FROM `#__users` as a';
		$query	.= '  inner join `#__easyblog_users` as b';
		$query	.= '    on a.id = b.id';
		$query	.= '  inner join `#__easyblog_likes` as c';
		$query	.= '    on a.id = c.created_by';
		$query	.= ' where c.content_id = ' . $db->Quote($contentId);
		$query	.= ' and c.`type` = '. $db->Quote($type);
		$query  .= ' order by c.id desc';

		$db->setQuery($query);
		$list   = $db->loadObjectList();

		if(count($list) <= 0)
		{
			return '';
		}

		// else continue here
		$onwerInside    = false;

		$names  = array();
		for($i = 0; $i < count($list); $i++)
		{

			if($list[$i]->user_id == $userId)
			{
				$onwerInside    = true;
				array_unshift($names, JText::_('COM_EASYBLOG_YOU') );
			}
			else
			{
				$blogger	= EasyBlogHelper::getTable( 'Profile', 'Table');
				$blogger->load( $list[ $i ]->user_id  );
				$names[]    = '<a href="' . $blogger->getProfileLink() . '">' . $list[$i]->displayname . '</a>';
			}
		}

		$max    = 3;
		$total  = count($names);
		$break	= 0;

		if($total == 1)
		{
			$break  = $total;
		}
		else
		{
			if($max >= $total)
			{
				$break  = $total - 1;
			}
			else if($max < $total)
			{
				$break  = $max;
			}
		}

		$main   = array_slice($names, 0, $break);
		$remain = array_slice($names, $break);

		$stringFront    = implode(", ", $main);
		$returnString   = '';

		if(count($remain) > 1)
		{
			$returnString   = JText::sprintf('COM_EASYBLOG_AND_OTHERS_LIKE_THIS', $stringFront, count($remain));
		}
		else if(count($remain) == 1)
		{
			$returnString   = JText::sprintf('COM_EASYBLOG_AND_LIKE_THIS', $stringFront, $remain[0]);
		}
		else
		{
			if( EasyBlogHelper::isLoggedIn() && $onwerInside )
			{
				$returnString   = JText::sprintf('COM_EASYBLOG_LIKE_THIS_SINGULAR', $stringFront);
			}
			else
			{
				$returnString   = JText::sprintf('COM_EASYBLOG_LIKE_THIS_PLURAL', $stringFront);
			}
		}

		return $returnString;
	}

	public static function dateTimePicker( $id, $caption, $date = '', $reset = false )
	{
		return '
			<div id="datetime_' . $id . '" class="datetime_container">
				<span class="datetime_caption">' . $caption . '</span>
				<a class="ui-button" href="javascript:void(0)" id="datetime_edit_' . $id . '" onclick="eblog.editor.datetimepicker.element(\''.$id.'\', \''.$reset.'\')">'.JText::_('COM_EASYBLOG_EDIT_BUTTON').'</a>
				<input type="hidden" name="' . $id . '" id="' . $id . '" value="'.$date.'" />
			</div>';
	}

	public static function formatBlogCommentsLite( $comments )
	{
		for($i = 0; $i < count($comments); $i++)
		{
			$row 		= $comments[$i];
			$creator 	= EasyBlogHelper::getTable( 'Profile', 'Table' );

			if($row->created_by != 0)
			{
				$creator->load($row->created_by);
				$user       = JFactory::getUser($row->created_by);
				$creator->setUser($user);
			}

			//get integrated avatar
			$row->poster		= $creator;
			$row->comment   	= EasyBlogCommentHelper::parseBBCode($row->comment);
		}

		return $comments;
	}

	/**
	 * Process trackback when saving blog post
	 * - check if there un-ping URLs
	 * - mark as sent if necessary
	 *
	 * @param	Integer		Blog ID
	 * @param	Array		Trackbacks from user input
	 * @param	Object		User object
	 */
	public static function processTrackbacks($blogId, $trackbacks, $user)
	{
		require_once( EBLOG_CLASSES . DS . 'trackback.php' );
		require_once( EBLOG_HELPERS . DS . 'router.php' );

		$trackbackTbl	= EasyBlogHelper::getTable( 'TrackbackSent' , 'Table' );

		$author		= EasyBlogHelper::getTable( 'Profile', 'Table' );
		$author->setUser( $user );


		for ( $x = 0; $x < count($trackbacks); $x++ )
		{
			// check if the URL has been added to our record
			$exists	= $trackbackTbl->load( $trackbacks[$x] , true , $blogId );

			// if not exists, we need to store them
			if( !$exists )
			{
				$trackbackTbl->post_id	= $blog->id;
				$trackbackTbl->url		= $url;
				$trackbackTbl->sent		= 0;
				$trackbackTbl->store();
			}
		}

		// now load trackback model
		$trackbackModel = self::getModel('TrackbackSent');

		// get lists of trackback URLs based on blog ID
		$tbacks = $trackbackModel->getSentTrackbacks( $blogId, true );

		// loop each URL, ping if necessary
		for( $x = 0; $x < count($tbacks); $x++ )
		{
			$tb		= new EasyBlogTrackBack( $author->getName() , $author->getName() , 'UTF-8');
			$text	= empty( $blog->intro ) ? $blog->content : $blog->intro;
			if( $tb->ping( $tbacks->url , EasyBlogRouter::getRoutedURL( 'index.php?option=com_easyblog&view=entry&id=' . $blog->id , false , true ) , $blog->title , $text ) )
			{
				//@task: Since the trackback was successful, store the trackback into the table.

				$trackbackTbl->load($tbacks->id);

				$new_trackbacks = array();
				$new_trackbacks['url']		= $tback->url;
				$new_trackbacks['post_id']	= $tback->post_id;
				$new_trackbacks['sent']		= 1;

				$trackbackTbl->bind($new_trackbacks);

				$trackbackTbl->store();
			}
		}
	}

	public static function getPageTitle($default='')
	{
		$itemid = JRequest::getVar('Itemid', '');

		if(empty($itemid))
		{
			return $default;
		}

		$menu		= JFactory::getApplication()->getMenu();
		$item		= $menu->getItem($itemid);

		// @task: If menu item cannot be found anywhere, just use the default
		if( !$item )
		{
			return $default;
		}

		$menuparams	= new JParameter( $item->params );
		$pageTitle	= $menuparams->get('page_title', $default);

		return $pageTitle;
	}

	public static function loadHeaders()
	{
		static $loaded = false;

		if( !$loaded )
		{
			if( EasyBlogHelper::getJoomlaVersion() >= '1.6' )
			{
				$uri 		= JFactory::getURI();
				$language	= $uri->getVar( 'lang' , 'none' );
				$app		= JFactory::getApplication();
				$config		= JFactory::getConfig();
				$router		= $app->getRouter();
				$url		= rtrim( JURI::root() , '/' ) . '/index.php?option=com_easyblog&lang=' . $language;

				if( $router->getMode() == JROUTER_MODE_SEF && JPluginHelper::isEnabled("system","languagefilter") )
				{
					$rewrite	= $config->get('sef_rewrite');

					$base		= str_ireplace( JURI::root( true ) , '' , $uri->getPath() );
					$path		=  $rewrite ? $base : JString::substr( $base , 10 );
					$path		= JString::trim( $path , '/' );
					$parts		= explode( '/' , $path );

					if( $parts )
					{
						// First segment will always be the language filter.
						$language	= reset( $parts );
					}
					else
					{
						$language	= 'none';
					}

					if( $rewrite )
					{
						$url		= rtrim( JURI::root() , '/' ) . '/' . $language . '/?option=com_easyblog';
						$language	= 'none';
					}
					else
					{
						$url		= rtrim( JURI::root() , '/' ) . '/index.php/' . $language . '/?option=com_easyblog';
					}
				}
			}
			else
			{

				$url		= rtrim( JURI::root() , '/' ) . '/index.php?option=com_easyblog';
			}

			$menu = JFactory::getApplication()->getmenu();

			$item = $menu->getActive();
			if( isset( $item->id) )
			{
				$url    .= '&Itemid=' . $item->id;
			}

			// Some SEF components tries to do a 301 redirect from non-www prefix to www prefix.
			// Need to sort them out here.
			$currentURL		= isset( $_SERVER[ 'HTTP_HOST' ] ) ? $_SERVER[ 'HTTP_HOST' ] : '';

			if( !empty( $currentURL ) )
			{
				// When the url contains www and the current accessed url does not contain www, fix it.
				if( stristr($currentURL , 'www' ) === false && stristr( $url , 'www') !== false )
				{
					$url	= str_ireplace( 'www.' , '' , $url );
				}

				// When the url does not contain www and the current accessed url contains www.
				if( stristr( $currentURL , 'www' ) !== false && stristr( $url , 'www') === false )
				{
					$url	= str_ireplace( '://' , '://www.' , $url );
				}
			}

			$config 		= EasyBlogHelper::getConfig();
			$lightboxTitle 	= $config->get( 'main_media_lightbox_title' ) ? 'true' : 'false';

			$document	= JFactory::getDocument();
			$ajaxData	=
"/*<![CDATA[*/
	var eblog_site 	= '" . $url . "';
	var spinnerPath = '" . EBLOG_SPINNER . "';
	var lang_direction	= '" . $document->direction . "';
	var eblog_lightbox_title = " . $lightboxTitle . ";
/*]]>*/";
			require_once( EBLOG_HELPERS . DS . 'loader.php' );


			$document->addScriptDeclaration( $ajaxData );

			$ebconfig = EasyBlogHelper::getConfig();
			if ($ebconfig->get( 'debug_javascript' ) || JRequest::getVar('ebjsdebug')==1)
			{
				EjaxLoader::_('jquery,jquery.ui.core,jquery.ui.widget,jquery.ui.position,jquery.ui.stars,jquery.tmpl,jquery.tmplPlus,jquery.hoverIntent,jquery.cookie,jquery.numeric,jquery.uid,jquery.checked,jquery.stackSelect,jquery.stackTip,jquery.stackSuggest,jquery.stretchToFit,jquery.fancybox,ejax-dev,eblog-dev','js');
			} else {
				EjaxLoader::_('ej,ejax,eblog','js');
			}

			EasyBlogHelper::addTemplateCss( 'common.css' );

			$loaded		= true;
		}
		return $loaded;
	}

	/**
	 * Get Tweetmeme and Google Buzz social button to show
	 *
	 * @params	Object		The blog item
	 *
	 * @return	String		HTML code to be shown
	 */
	public static function showSocialButton( $blog , $frontpage = false )
	{
		$config 	= EasyBlogHelper::getConfig();
		$document   = JFactory::getDocument();

		// get all required settings
		$tweetmeme	= $config->get('main_tweetmeme') && !$frontpage || $frontpage && $config->get('main_tweetmeme_frontpage' , $config->get('social_show_frontpage') ) && $config->get('main_tweetmeme');	// TweetMeme

		$isBottom   = false;

		// RTL enabled?
		$rtl			= $config->get('social_rtl', 1);
		$extraStyling   = '';

		// get prefered social button position
		$pos 		= $config->get('main_socialbutton_position' , 'left' );
		$isBottom   = ($pos == 'bottom' || $pos == 'top') ? true : false;

		$extraStyling   .= ($pos == 'top') ? ' top' : '';
		$extraStyling   .= ($pos == 'bottom') ? ' bottom' : '';

		// in here, if the possition is bottom, we treat it as left for the css styling later.
		$pos        = ($pos == 'bottom' || $pos == 'top') ? 'left' : $pos;

		// if RTL setting is enabled
		if ( $rtl )
		{
			// only process this if the direction is RTK
			if ( $document->direction == 'rtl' )
			{
				// if user set position to left, we change it to right.
				// and the other way around too
				if ( $pos == 'left' )
				{
					$pos = 'right';
				}
				else
				{
					$pos = 'left';
				}
			}
		}

		$teamIdLink = '';
		if( isset( $row->team ) )
			$teamIdLink = '&team=' . $row->team;

		$html 				= '';
		$tweetmemeHTML 		= '';
		$twitterHTML    	= '';
		$linkedinHTML		= '';
		$stumbleHTML		= '';
		$diggHTML 			= '';

		$loadsocialbutton	= true;

		if($config->get('main_password_protect') && !empty($blog->blogpassword))
		{
			if(!EasyBlogHelper::verifyBlogPassword($blog->blogpassword, $blog->id))
			{
				$loadsocialbutton = false;
			}
		}

		if($loadsocialbutton)
		{
			$extraStyling	.= ( $isBottom ) ? ' width-full' : '';

			// Tweetmeme Integrations
			if ( $tweetmeme )
			{
				require_once( EBLOG_CLASSES . DS . 'tweetmeme.php' );

				$tweetmemeHTML	= EasyBlogTweetmeme::getHTML( $blog );
				if($isBottom)
				{
					$tweetmemeHTML  = '<div class="socialbutton-vertical align' . $pos . '">'.$tweetmemeHTML.'</div>';
				}
			}

			$dataURL		= EasyBlogRouter::getRoutedURL('index.php?option=com_easyblog&view=entry&id=' . $blog->id . $teamIdLink, false, true);
			$dataTitle		= urlencode( $blog->title );

			$linkedinButton	= !$frontpage && $config->get( 'main_linkedin_button' ) || $frontpage && $config->get( 'main_linkedin_button_frontpage' , $config->get('social_show_frontpage') )  && $config->get( 'main_linkedin_button' );
			if( $linkedinButton )
			{
				$style	=  $config->get( 'main_linkedin_button_style' );

				switch( $style )
				{
					case 'vertical':
						$counter	= ' data-counter="top"';
					break;
					case 'horizontal':
						$counter	= ' data-counter="right"';
					break;
					default:
						$counter	= '';
					break;
				}
				$linkedinHTML	= '<div class="social-button linkedin-share"><script type="text/javascript" src="https://platform.linkedin.com/in.js"></script><script type="in/share" data-url="' . $dataURL . '"' . $counter . '></script></div>';

				if($isBottom)
				{
					$linkedinHTML  = '<div class="socialbutton-vertical align' . $pos . '">'.$linkedinHTML.'</div>';
				}
			}

			$diggButton		= !$frontpage && $config->get( 'main_digg_button' ) || $frontpage && $config->get( 'main_digg_button_frontpage' , $config->get('social_show_frontpage') ) && $config->get( 'main_digg_button');

			if( $diggButton )
			{
				$diggHTML	= '';

				switch( $config->get( 'main_digg_button_style' ) )
				{
					case 'compact':
						$class 	= 'DiggCompact';
					break;
					case 'medium':
					default:
						$class 	= 'DiggMedium';
					break;
				}

				$diggHTML	= '<script type="text/javascript">(function() {
				var s = document.createElement(\'SCRIPT\'), s1 = document.getElementsByTagName(\'SCRIPT\')[0];
				s.type = \'text/javascript\';
				s.async = true;
				s.src = \'http://widgets.digg.com/buttons.js\';
				s1.parentNode.insertBefore(s, s1);
				})();
				</script>
				<!-- Wide Button -->
				<div class="social-button digg-share"><a class="DiggThisButton ' . $class . '" href="https://digg.com/submit?url=' . $dataURL . '&amp;title=' . $dataTitle . '"></a></div>';

				if( $isBottom )
				{
					$diggHTML  = '<div class="socialbutton-vertical align' . $pos . '">'.$diggHTML.'</div>';
				}
			}

			$stumbleButton	= !$frontpage && $config->get( 'main_stumbleupon_button' ) || $frontpage && $config->get( 'main_stumbleupon_button_frontpage', $config->get('social_show_frontpage')) && $config->get( 'main_stumbleupon_button' );

			if( $stumbleButton )
			{
				$style	=  $config->get( 'main_stumbleupon_button_style' );

				switch( $style )
				{
					case 'vertical':
						$counter	= '5';
					break;
					case 'horizontal':
						$counter	= '1';
					break;
					default:
						$counter	= '6';
					break;
				}
				$stumbleHTML	= '<div class="social-button stumbleupon-share"><script src="https://www.stumbleupon.com/hostedbadge.php?s=' . $counter . '&r=' . $dataURL . '"></script></div>';

				if($isBottom)
				{
					$stumbleHTML  = '<div class="socialbutton-vertical align' . $pos . '">'.$stumbleHTML.'</div>';
				}
			}

			$twitterButton	= ( !$frontpage && $config->get('main_twitter_button') ) || ($frontpage && $config->get('main_twitter_button_frontpage', $config->get('social_show_frontpage')) && $config->get('main_twitter_button'));
			if( $twitterButton )
			{
				$style			= $config->get( 'main_twitter_button_style' );
				$dataURL		= EasyBlogRouter::getRoutedURL('index.php?option=com_easyblog&view=entry&id=' . $blog->id . $teamIdLink, false, true);
				$twitterHTML	= '<div class="social-button retweet"><a href="https://twitter.com/share" class="twitter-share-button" data-url="' . $dataURL . '" data-counturl="'.$dataURL.'" data-count="' . $style .'">Tweet</a><script type="text/javascript" src="https://platform.twitter.com/widgets.js"></script></div>';
				if($isBottom)
				{
					$twitterHTML  = '<div class="socialbutton-vertical align' . $pos . '">'.$twitterHTML.'</div>';
				}
			}

			$fblayout			= $config->get('main_facebook_like_layout');
			$fbLikesButton		= $config->get('main_facebook_like');
			$facebookLikeHTML   = '';

			if($fblayout != 'standard')
			{
				//if the layout is box_count then only we stick the fb likes with others social button.
				require_once( EBLOG_CLASSES . DS . 'facebook.php' );
				$facebookLikeHTML	= EasyBlogFacebookLikes::getLikeHTML( $blog );

				$facebookLikeHTML	= '<div class="social-button facebook-like">' . $facebookLikeHTML . '</div>';
				if($isBottom)
				{
					$facebookLikeHTML  = '<div class="socialbutton-vertical align' . $pos . '">'.$facebookLikeHTML.'</div>';
				}
			}

			// Google +1
			$googlePlusOne	= ( !$frontpage && $config->get('main_googleone') ) || ( $frontpage && $config->get( 'main_googleone_frontpage', $config->get('social_show_frontpage') ) && $config->get('main_googleone'));
			$socialFrontEnd	= $config->get( 'main_googleone_frontpage', 0 );
			$googleHTML     = '';
			if( $googlePlusOne )
			{
				$size		= $config->get('main_googleone_layout');
				$dataURL	= EasyBlogRouter::getRoutedURL('index.php?option=com_easyblog&view=entry&id=' . $blog->id . $teamIdLink, false, true);


				// Add snippet info into headers
				$document	= JFactory::getDocument();

				$meta	= EasyBlogHelper::getTable( 'Meta' , 'Table' );
				$meta->loadByType( META_TYPE_POST , $blog->id );

				$document->addCustomTag( '<meta itemprop="name" content="' . $blog->title . '" />' );

				if( !empty( $meta->description ) )
				{
					$document->addCustomTag( '<meta itemprop="description" content="' . $meta->description . '" />' );
				}
				else
				{
					$maxContentLen	= 350;
					$text			= strip_tags( $blog->intro . $blog->content );
					$text			= ( JString::strlen( $text ) > $maxContentLen ) ? JString::substr( $text, 0, $maxContentLen) . '...' : $text;
					$document->addCustomTag( '<meta itemprop="description" content="' . $text . '" />' );
				}

				$image	= EasyBlogHelper::getFirstImage( $blog->intro . $blog->content );

				if( $image !== false )
				{
					$document->addCustomTag( '<meta itemprop="image" content="' . $image . '" />' );
				}

				$googleHTML	.= '<div class="social-button google-plusone">';
				if( ! $socialFrontEnd )
					$googleHTML	.= '<script type="text/javascript" src="https://apis.google.com/js/plusone.js"></script>';
				$googleHTML	.= '<g:plusone size="' . $size . '" href="' . $dataURL . '"></g:plusone>';
				$googleHTML	.= '</div>';

				if($isBottom)
				{
					$googleHTML	= '<div class="socialbutton-vertical align' . $pos . '">' . $googleHTML . '</div>';
				}
			}

			// only generate output if both or either one is enabled
			if ( $tweetmeme OR $twitterButton OR $fbLikesButton OR $googleHTML OR $linkedinHTML OR $stumbleHTML OR $diggHTML )
			{
				$html = '<div id="socialbutton" class="align' . $pos . $extraStyling . '">' . $tweetmemeHTML . $linkedinHTML . $stumbleHTML . $diggHTML . $twitterHTML . $googleHTML . $facebookLikeHTML . '</div>';
				$html .= ($isBottom) ? '<div style="clear:both;"></div>' : '';
			}

		}

		echo $html;
	}

	// this function used to show the login form
	public static function showLogin($return='')
	{
		$my = JFactory::getUser();

		if($my->id == 0)
		{

			if( EasyBlogHelper::getJoomlaVersion() >= '1.6' ) {
				$comUserOption	= 'com_users';
				$tasklogin		= 'user.login';
				$tasklogout		= 'user.logout';
				$viewRegister	= 'registration';
				$inputPassword	= 'password';
			} else {
				$comUserOption	= 'com_user';
				$tasklogin		= 'login';
				$tasklogout		= 'logout';
				$viewRegister	= 'register';
				$inputPassword	= 'passwd';
			}

			if(empty($return))
			{
				$currentUri = JRequest::getURI();
				$uri		= base64_encode($currentUri);
			}
			else
			{
				$uri		= $return;
			}


			$tpl	= new CodeThemes();
			$tpl->set( 'return' , $uri );

			$tpl->set( 'comUserOption' , $comUserOption );
			$tpl->set( 'tasklogin' , $tasklogin );
			$tpl->set( 'tasklogout' , $tasklogout );
			$tpl->set( 'viewRegister' , $viewRegister );
			$tpl->set( 'inputPassword' , $inputPassword );

			echo $tpl->fetch( 'guest.login.php' );
		}
	}


	public static function getThemeInfo( $name )
	{
		jimport( 'joomla.filesystem.file' );

		$mainframe	= JFactory::getApplication();

		$file =	'';

		// We need to specify if the template override folder also have config.ini file
		if ( JFile::exists( JPATH_ROOT . DS . 'templates' . DS . $mainframe->getTemplate() . DS . 'html' . DS . 'com_easyblog' . DS . 'config.ini' ) )
		{
			$file = JPATH_ROOT . DS . 'templates' . DS . $mainframe->getTemplate() . DS . 'html' . DS . 'com_easyblog' . DS . 'config.ini';
		}

		// then check the current theme folder
		elseif ( JFile::exists( JPATH_ROOT . DS . 'components' . DS . 'com_easyblog'. DS . 'themes' . DS . $name . DS . 'config.ini' ) )
		{
			$file = JPATH_ROOT . DS . 'components' . DS . 'com_easyblog' . DS . 'themes' . DS . $name . DS . 'config.ini';
		}

		if( !empty( $file ) )
		{
			$raw	= JFile::read( $file );
			$info	= new JParameter( $raw );

			return $info;
		}

		return new JParameter( '' );
	}


	/**
	 * parentId - if this option spcified, it will list the parent and all its childs categories.
	 * userId - if this option specified, it only return categories created by this userId
	 * outType - the output type. Currently supported links and drop down selection
	 * eleName - the element name of this populated categeries provided the outType os dropdown selection.
	 * default - the default value. If given, it used at dropdown selection (auto select)
	 * isWrite - determine whether the categories list used in write new page or not.
	 * isPublishedOnly - if this option is true, only published categories will fetched.
	 */

	public static function populateCategories($parentId, $userId, $outType, $eleName, $default, $isWrite = false, $isPublishedOnly = false, $isFrontendWrite = false)
	{

		JModel::addIncludePath(JPATH_ROOT.DS.'components'.DS.'com_easyblog'.DS.'models');
		if(class_exists('EasyBlogModelCategories'))
		{
			$catModel   = new EasyBlogModelCategories();
		}
		else
		{
			require_once( JPATH_ROOT.DS.'components'.DS.'com_easyblog'.DS.'models'.DS.'categories.php' );
			$catModel   = new EasyBlogModelCategories();
		}

		$parentCat	= null;

		if(! empty($userId))
		{
			$parentCat  = $catModel->getParentCategories($userId, 'blogger', $isPublishedOnly, $isFrontendWrite);
		}
		else if(! empty($parentId))
		{
			$parentCat  = $catModel->getParentCategories($parentId, 'category', $isPublishedOnly, $isFrontendWrite);
		}
		else
		{
			$parentCat  = $catModel->getParentCategories('', 'all', $isPublishedOnly, $isFrontendWrite);
		}

		$ignorePrivate  = false;

		switch($outType)
		{
			case 'link' :
				$ignorePrivate  = false;
				break;
			case 'popup':
			case 'select':
			default:
				$ignorePrivate  = true;
				break;
		}

		if(! empty($parentCat))
		{
			for($i = 0; $i < count($parentCat); $i++)
			{
				$parent = $parentCat[$i];

				//reset
				$parent->childs = null;

				EasyBlogHelper::buildNestedCategories($parent->id, $parent, $ignorePrivate, $isPublishedOnly, $isFrontendWrite);
			}//for $i
		}//end if !empty $parentCat

		if( $isWrite )
		{
			$defaultCatId   = EasyBlogHelper::getDefaultCategoryId();
			$default    = ( empty( $default ) ) ? $defaultCatId : $default;
		}

		$formEle    = '';

		if( $outType == 'select' && $isWrite )
		{
			$selected	= !$default ? ' selected="selected"' : '';
			$formEle	.= '<option value="0"' . $selected . '>' . JText::_( 'COM_EASYBLOG_SELECT_A_CATEGORY' ) . '</option>';
		}

		if( $parentCat )
		{
			foreach($parentCat as $category)
			{
				if($outType == 'popup')
				{
					$formEle   .= '<div class="category-list-item" id="'.$category->id.'"><a href="javascript:void(0);" onclick="eblog.dashboard.selectCategory(\''. $category->id. '\')">' .$category->title . '</a>';
					$formEle   .= '<input type="hidden" id="category-list-item-'.$category->id.'" value="'.$category->title.'" />';
					$formEle   .= '</div>';
				}
				else
				{
					$selected   = ($category->id == $default) ? ' selected="selected"' : '';
					$formEle   .= '<option value="'.$category->id.'" ' . $selected. '>' . JText::_( $category->title ) . '</option>';
				}

				EasyBlogHelper::accessNestedCategories($category, $formEle, '0', $default, $outType);
			}
		}

		$html   = '';
		$html	.= '<select name="' . $eleName . '" id="' . $eleName .'" class="inputbox">';
		if(! $isWrite)
			$html	.=	'<option value="0">' . JText::_('COM_EASYBLOG_SELECT_PARENT_CATEGORY') . '</option>';
		$html	.=	$formEle;
		$html	.= '</select>';

		return $html;
	}


	public static function buildNestedCategories($parentId, $parent, $ignorePrivate = false, $isPublishedOnly = false, $isWrite = false)
	{
		$my     = JFactory::getUser();

		JModel::addIncludePath(JPATH_ROOT.DS.'components'.DS.'com_easyblog'.DS.'models');
		if(class_exists('EasyBlogModelCategories'))
		{
			$catModel   = new EasyBlogModelCategories();
		}
		else
		{
			require_once( JPATH_ROOT.DS.'components'.DS.'com_easyblog'.DS.'models'.DS.'categories.php' );
			$catModel   = new EasyBlogModelCategories();
		}

		$childs 			= $catModel->getChildCategories($parentId, $isPublishedOnly, $isWrite);
		$accessibleCatsIds 	= EasyBlogHelper::getAccessibleCategories( $parentId );

		if(! empty($childs))
		{
			for($j = 0; $j < count($childs); $j++)
			{
				$child  = $childs[$j];
				$child->childs = null;

				if(! $ignorePrivate)
				{
					if( count( $accessibleCatsIds ) > 0)
					{
						$access = false;
						foreach( $accessibleCatsIds as $canAccess)
						{
							if( $canAccess->id == $child->id)
							{
								$access = true;
							}
						}

						if( !$access )
							continue;

					}
					else
					{
						continue;
					}
				}

				if(! EasyBlogHelper::buildNestedCategories($child->id, $child, $ignorePrivate, $isPublishedOnly))
				{
					$parent->childs[]   = $child;
				}
			}// for $j
		}
		else
		{
			return false;
		}

	}


	public static function accessNestedCategories($arr, &$html, $deep='0', $default='0', $type='select', $linkDelimiter = '')
	{
		if(isset($arr->childs) && is_array($arr->childs))
		{
			$sup    = '<sup>|_</sup>';
			$space  = '';
			$ld     = (empty($linkDelimiter)) ? '>' : $linkDelimiter;

			if($type == 'select' || $type == 'popup')
			{
				$deep++;
				for($d=0; $d < $deep; $d++)
				{
					$space .= '&nbsp;&nbsp;&nbsp;';
				}
			}

			for($j	= 0; $j < count($arr->childs); $j++)
			{
				$child  = $arr->childs[$j];

				if($type == 'select')
				{
					$selected   = ($child->id == $default) ? ' selected="selected"' : '';

					$html   .= '<option value="'.$child->id.'" ' . $selected . '>' . $space . $sup . $child->title . '</option>';
				}
				else if($type == 'popup')
				{
					$html   .= '<div class="category-list-item" id="'.$child->id.'">' . $space . $sup . '<a href="javascript:void(0);" onclick="eblog.dashboard.selectCategory(\''. $child->id. '\')">' .$child->title . '</a>';
					$html   .= '<input type="hidden" id="category-list-item-'.$child->id.'" value="'.$child->title.'" />';
					$html   .= '</div>';
				}
				else
				{
					$str    = '<a href="' . EasyBlogRouter::_('index.php?option=com_easyblog&view=categories&layout=listings&id='.$child->id) . '">' . $child->title . '</a>';
					$html   .= (empty($html)) ? $str : $ld . $str;
				}

				EasyBlogHelper::accessNestedCategories($child, $html, $deep, $default, $type, $linkDelimiter);
			}
		}
		else
		{
			return false;
		}
	}



	public static function accessNestedCategoriesId($arr, &$newArr)
	{
		if(isset($arr->childs) && is_array($arr->childs))
		{

			for($j	= 0; $j < count($arr->childs); $j++)
			{
				$child  = $arr->childs[$j];

				$newArr[]   = $child->id;
				EasyBlogHelper::accessNestedCategoriesId($child, $newArr);
			}
		}
		else
		{
			return false;
		}
	}


	/**
	 * function to retrieve the linkage backward from a child id.
	 * return the full linkage from child up to parent
	 */

	public static function populateCategoryLinkage($childId)
	{
		$arr        = array();
		$category	= EasyBlogHelper::getTable( 'ECategory' , 'Table' );
		$category->load($childId);

		$obj    = new stdClass();
		$obj->id 	= $category->id;
		$obj->title = $category->title;
		$obj->alias = $category->alias;

		$arr[]  = $obj;

		if((!empty($category->parent_id)))
		{
			EasyBlogHelper::accessCategoryLinkage($category->parent_id, $arr);
		}

		$arr    = array_reverse($arr);
		return $arr;

	}

	public static function accessCategoryLinkage($childId, &$arr)
	{
		$category	= EasyBlogHelper::getTable( 'ECategory' , 'Table' );
		$category->load($childId);

		$obj    = new stdClass();
		$obj->id 	= $category->id;
		$obj->title = $category->title;
		$obj->alias = $category->alias;



		$arr[]  = $obj;

		if((!empty($category->parent_id)))
		{
			EasyBlogHelper::accessCategoryLinkage($category->parent_id, $arr);
		}
		else
		{
			return false;
		}
	}


	/**
	 * Get post title by ID
	 */
	public static function getPostTitle($id)
	{
		$db = JFactory::getDBO();

		$query = 'SELECT ' . $db->nameQuote('title') . ' FROM ' . $db->nameQuote('#__easyblog_post') . ' WHERE id = ' . $db->Quote($id);
		$db->setQuery($query);
		return $db->loadResult();
	}


	/**
	 * Check AlphaUserPoints Integration
	 */
	public static function isAUPEnabled()
	{
		jimport('joomla.filesystem.file');

		$config = EasyBlogHelper::getConfig();
		$aup	= JPATH_ROOT . DS . 'components' . DS . 'com_alphauserpoints' . DS . 'helper.php';

		// make sure the config is enabled
		if ( $config->get('main_alpha_userpoint') ) {

			if ( JFile::exists( $aup ) )
			{
				require_once( $aup );
				return true;
			}
			else
			{
				return false;
			}
		}
	}

	public static function storeSession($data, $key, $ns = 'COM_EASYBLOG')
	{
		$mySess	= JFactory::getSession();
		$mySess->set($key, $data, $ns);
	}

	public static function getSession($key, $ns = 'COM_EASYBLOG')
	{
		$data   = null;

		$mySess = JFactory::getSession();
		if($mySess->has($key, $ns))
		{
			$data   = $mySess->get($key, '', $ns);
			$mySess->clear($key, $ns);
			return $data;
		}
		else
		{
			return $data;
		}
	}

	public static function clearSession($key, $ns = 'COM_EASYBLOG')
	{
		$mySess = JFactory::getSession();
		if($mySess->has($key, $ns))
		{
			$mySess->clear($key, $ns);
		}
		return true;
	}

	public static function isTeamBlogJoined($userId, $teamId)
	{
		$teamIds	= EasyBlogHelper::getViewableTeamIds($userId);
		return in_array($teamId, $teamIds);
	}

	// this function used to show the access denied page
	public static function showAccessDenied($type='', $access = '0')
	{
		$message    = JText::_('COM_EASYBLOG_SORRY_YOU_DO_NOT_HAVE_PERMISSION_TO_VIEW');

		if($type == 'teamblog')
		{
			if($access == '1')
			{
				$message    = JText::_('COM_EASYBLOG_TEAMBLOG_MEMBERS_ONLY');
			}
		}


		$tpl	= new CodeThemes();
		$tpl->set( 'message' , $message );
		echo $tpl->fetch( 'access.denied.php' );
	}

	public static function getJoomlaVersion()
	{
		$jVerArr   = explode('.', JVERSION);
		$jVersion  = $jVerArr[0] . '.' . $jVerArr[1];


		return $jVersion;
	}

	/**
	 * Used in J1.6!. To retrieve list of superadmin users's id.
	 * array
	 */

	public static function getSAUsersIds()
	{
		$db = JFactory::getDBO();

		$query  = 'SELECT a.`id`, a.`title`';
		$query	.= ' FROM `#__usergroups` AS a';
		$query	.= ' LEFT JOIN `#__usergroups` AS b ON a.lft > b.lft AND a.rgt < b.rgt';
		$query	.= ' GROUP BY a.id';
		$query	.= ' ORDER BY a.lft ASC';

		$db->setQuery($query);
		$result = $db->loadObjectList();

		$saGroup    = array();
		foreach($result as $group)
		{
			if(JAccess::checkGroup($group->id, 'core.admin'))
			{
				$saGroup[]  = $group;
			}
		}


		//now we got all the SA groups. Time to get the users
		$saUsers    = array();
		if(count($saGroup) > 0)
		{
			foreach($saGroup as $sag)
			{
				$userArr	= JAccess::getUsersByGroup($sag->id);
				if(count($userArr) > 0)
				{
					foreach($userArr as $user)
					{
						$saUsers[]    = $user;
					}
				}
			}
		}

		return $saUsers;
	}

	public static function getDefaultSAIds()
	{
		$saUserId   = '62';

		if(EasyBlogHelper::getJoomlaVersion() >= '1.6')
		{
			$saUsers    = EasyBlogHelper::getSAUsersIds();
			$saUserId   = $saUsers[0];
		}

		return $saUserId;
	}

	/*
	 * Method for broswer detection
	 */
	private static function getBrowserUserAgent()
	{
		$browser = new stdClass;

		// set to lower case to avoid errors, check to see if http_user_agent is set
		$navigator_user_agent = ( isset( $_SERVER['HTTP_USER_AGENT'] ) ) ? strtolower( $_SERVER['HTTP_USER_AGENT'] ) : '';

		// run through the main browser possibilities, assign them to the main $browser variable
		if (stristr($navigator_user_agent, "opera"))
		{
				$browser->userAgent = 'opera';
				$browser->dom = true;
		}
		elseif (stristr($navigator_user_agent, "msie 8"))
		{
				$browser->userAgent = 'msie8';
				$browser->dom = false;
		}
		elseif (stristr($navigator_user_agent, "msie 7"))
		{
				$browser->userAgent = 'msie7';
				$browser->dom = false;
		}
		elseif (stristr($navigator_user_agent, "msie 4"))
		{
				$browser->userAgent = 'msie4';
				$browser->dom = false;
		}
		elseif (stristr($navigator_user_agent, "msie"))
		{
				$browser->userAgent = 'msie';
				$browser->dom = true;
		}
		elseif ((stristr($navigator_user_agent, "konqueror")) || (stristr($navigator_user_agent, "safari")))
		{
				$browser->userAgent = 'safari';
				$browser->dom = true;
		}
		elseif (stristr($navigator_user_agent, "gecko"))
		{
				$browser->userAgent = 'mozilla';
				$browser->dom = true;
		}
		elseif (stristr($navigator_user_agent, "mozilla/4"))
		{
				$browser->userAgent = 'ns4';
				$browser->dom = false;
		}
		else
		{
				$browser->dom = false;
				$browser->userAgent = 'Unknown';
		}

		return $browser;
	}

	public static function getPDFlinkProperties()
	{
		$browser = EasyBlogHelper::getBrowserUserAgent();

		switch($browser->userAgent)
		{
			case 'msie8':
			case 'msie7':
				$properties = ' target="_BLANK" ';
				break;
			default:
				$properties = ' onclick="window.open(this.href,\'win2\',\'status=no,toolbar=no,scrollbars=yes,titlebar=no,menubar=no,resizable=yes,width=640,height=480,directories=no,location=no\'); return false;" ';
		}

		return $properties;
	}

	public static function getFirstImage( $content )
	{
		//try to search for the 1st img in the blog
		$img            = '';
		$pattern		= '#<img[^>]*>#i';
		preg_match( $pattern , $content , $matches );

		if( $matches )
		{
			$img    = $matches[0];
		}


		//image found. now we process further to get the absolute image path.
		if(! empty($img) )
		{
			//get the img src

			$pattern = '/src=[\"\']?([^\"\']?.*(png|jpg|jpeg|gif))[\"\']?/i';
			preg_match($pattern, $img, $matches);
			if($matches)
			{
				$imgPath   = $matches[1];
				$imgSrc    = EasyImageHelper::rel2abs($imgPath, JURI::root());

				return $imgSrc;
			}
		}

		return false;
	}

	public static function getFBInitScript()
	{
		$config 	= EasyBlogHelper::getConfig();
		$fbScript   = '';

		if($config->get('main_facebook_like'))
		{
			$fbScript = '<div id="fb-root"></div>';
			$fbScript .= '<script type="text/javascript">';
			$fbScript .= '	window.fbAsyncInit = function() {';
			$fbScript .= '		FB.init({appId: \'' . $config->get('main_facebook_like_admin') . '\', status: true, cookie: true,';
			$fbScript .= '			xfbml: true});';
			$fbScript .= '	};';
			$fbScript .= '	(function() {';
			$fbScript .= '		var e = document.createElement(\'script\'); e.async = true;';
			$fbScript .= '		e.src = document.location.protocol +';
			$fbScript .= '			\'//connect.facebook.net/en_US/all.js\';';
			$fbScript .= '		document.getElementById(\'fb-root\').appendChild(e);';
			$fbScript .= '	}());';
			$fbScript .= '</script>';
		}

		return $fbScript;
	}

	public static function getBloggerTheme()
	{
		$id		= EasyBlogRouter::isBloggerMode();

		if( empty( $id ) )
		{
			return false;
		}

		$profile = EasyBlogHelper::getTable('Profile', 'Table');
		$profile->load( $id );

		$userparams	= new JParameter( $profile->params );

		return $userparams->get('theme', false);
	}

	public static function getFeaturedImage( $content )
	{
		$pattern = '#<img class="featured[^>]*>#i';
		preg_match( $pattern , $content , $matches );

		if( isset( $matches[0] ) )
		{
			return $matches[0];
		}

		// If featured image is not supplied, try to use the first image as the featured post.
		$pattern				= '#<img[^>]*>#i';

		preg_match( $pattern , $content , $matches );

		if( isset( $matches[0] ) )
		{
			return $matches[0];
		}

		// If all else fail, try to use the default image
		return false;
	}

	public static function getJoomlaUserGroups( $cid = '' )
	{
		$db = JFactory::getDBO();

		if(EasyBlogHelper::getJoomlaVersion() >= '1.6')
		{
			$query = 'SELECT a.id, a.title AS `name`, COUNT(DISTINCT b.id) AS level';
			$query .= ' , GROUP_CONCAT(b.id SEPARATOR \',\') AS parents';
			$query .= ' FROM #__usergroups AS a';
			$query .= ' LEFT JOIN `#__usergroups` AS b ON a.lft > b.lft AND a.rgt < b.rgt';
		}
		else
		{
			$query	= 'SELECT `id`, `name`, 0 as `level` FROM ' . $db->nameQuote('#__core_acl_aro_groups') . ' a ';
		}

		// condition
		$where  = array();

		// we need to filter out the ROOT and USER dummy records.
		if(EasyBlogHelper::getJoomlaVersion() < '1.6')
		{
			$where[] = '(a.`id` > 17 AND a.`id` < 26)';
		}

		if( !empty( $cid ) )
		{
			$where[] = ' a.`id` = ' . $db->quote($cid);
		}
		$where = ( count( $where ) ? ' WHERE ' .implode( ' AND ', $where ) : '' );

		$query  .= $where;

		// grouping and ordering
		if( EasyBlogHelper::getJoomlaVersion() >= '1.6' )
		{
			$query	.= ' GROUP BY a.id';
			$query	.= ' ORDER BY a.lft ASC';
		}
		else
		{
			$query 	.= ' ORDER BY a.id';
		}

		$db->setQuery( $query );
		$result = $db->loadObjectList();

		return $result;
	}

	public static function getUserGids( $userId = '' )
	{
		$user   = '';

		if( empty($userId) )
		{
			$user   = JFactory::getUser();
		}
		else
		{
			$user   = JFactory::getUser($userId);
		}

		if( EasyBlogHelper::getJoomlaVersion() >= '1.6' )
		{
			$groupIds = $user->groups;

			$grpId  = array();

			foreach($groupIds as $key => $val)
			{
				$grpId[] = $val;
			}

			return $grpId;
		}
		else
		{
			return array( $user->gid );
		}
	}

	public static function getAccessibleCategories( $parentId = 0 )
	{
		$db 			= JFactory::getDBO();
		$my 			= JFactory::getUser();

		$gids   	= '';
		$catQuery	= 	'select distinct a.`id`, a.`private`';
		$catQuery	.=  ' from `#__easyblog_category` as a';
		$catQuery	.=  ' where (a.`private` = ' . $db->Quote('0');

		if( EasyBlogHelper::getJoomlaVersion() >= '1.6' )
		{
			$gid    = array();
			if( $my->id == 0 )
			{
				$gid 	= JAccess::getGroupsByUser(0, false);
			}
			else
			{
				$gid 	= JAccess::getGroupsByUser($my->id, false);
			}

			if( count( $gid ) > 0 )
			{
				foreach( $gid as $id)
				{
					$gids   .= ( empty($gids) ) ? $db->Quote( $id ) : ',' . $db->Quote( $id );
				}
			}

			$catQuery   .= ' OR a.`id` IN (';
			$catQuery   .= '     SELECT c.category_id FROM `#__easyblog_category_acl` as c ';
			$catQuery   .= '        WHERE c.acl_id = ' .$db->Quote( CATEGORY_ACL_ACTION_VIEW );
			$catQuery   .= '        AND c.content_id IN (' . $gids . ') )';
			$catQuery   .= ')';
		}
		else
		{
			if( $my->id == 0 )
			{
				$catQuery   .= ')';
			}
			else
			{
				$gid	= EasyBlogHelper::getUserGids();

				if( count( $gid ) > 0 )
				{
					foreach( $gid as $id)
					{
						$gids   .= ( empty($gids) ) ? $db->Quote( $id ) : ',' . $db->Quote( $id );
					}


					$catQuery   .= ' OR a.`id` IN (';
					$catQuery   .= '     SELECT c.category_id FROM `#__easyblog_category_acl` as c ';
					$catQuery   .= '        WHERE c.acl_id = ' .$db->Quote( CATEGORY_ACL_ACTION_VIEW );
					$catQuery   .= '        AND c.content_id IN (' . $gids . ') )';
					$catQuery   .= ')';
				}
			}
		}

		$catQuery   .= ' AND a.parent_id = ' . $db->Quote($parentId);

		$db->setQuery($catQuery);
		$result = $db->loadObjectList();

		return $result;
	}

	public static function getPrivateCategories()
	{
		$db 			= JFactory::getDBO();
		$my 			= JFactory::getUser();
		$excludeCats    = array();

		$catQuery = '';
		// get all private categories id
		if($my->id == 0)
		{
			$catQuery	= 	'select distinct a.`id`, a.`private`';
			$catQuery	.=  ' from `#__easyblog_category` as a';
			$catQuery	.=	' 	left join `#__easyblog_category_acl` as b on a.`id` = b.`category_id` and b.`acl_id` = ' . $db->Quote( CATEGORY_ACL_ACTION_VIEW );
			$catQuery	.=  ' where a.`private` != ' . $db->Quote('0');

			$gid    = array();
			$gids   = '';

			if( EasyBlogHelper::getJoomlaVersion() >= '1.6' )
			{
				$gid 	= JAccess::getGroupsByUser(0, false);
			}
			else
			{
				$gid 	= EasyBlogHelper::getUserGids();
			}

			if( count( $gid ) > 0 )
			{
				foreach( $gid as $id)
				{
					$gids   .= ( empty($gids) ) ? $db->Quote( $id ) : ',' . $db->Quote( $id );
				}
				$catQuery   .= ' and b.`category_id` NOT IN (';
				$catQuery   .= '     SELECT c.category_id FROM `#__easyblog_category_acl` as c ';
				$catQuery   .= '        WHERE c.acl_id = ' .$db->Quote( CATEGORY_ACL_ACTION_VIEW );
				$catQuery   .= '        AND c.content_id IN (' . $gids . ') )';
			}

			// echo $catQuery;exit;
		}
		else
		{
			$gid	= EasyBlogHelper::getUserGids();
			$gids   = '';
			if( count( $gid ) > 0 )
			{
				foreach( $gid as $id)
				{
					$gids   .= ( empty($gids) ) ? $db->Quote( $id ) : ',' . $db->Quote( $id );
				}
			}

			$catQuery = 'select id from `#__easyblog_category` as a';
			$catQuery .= ' where not exists (';
			$catQuery .= '		select b.category_id from `#__easyblog_category_acl` as b';
			$catQuery .= '			where b.category_id = a.id and b.`acl_id` = '. $db->Quote( CATEGORY_ACL_ACTION_VIEW );
			$catQuery .= '			and b.type = ' . $db->Quote('group');
			$catQuery .= '			and b.content_id IN (' . $gids . ')';
			$catQuery .= '      )';
			$catQuery .= ' and a.`private` = ' . $db->Quote( CATEGORY_PRIVACY_ACL );
		}

		if( !empty( $catQuery ) )
		{

			$db->setQuery($catQuery);
			$result = $db->loadObjectList();

			for($i=0; $i < count($result); $i++)
			{
				$item   = $result[$i];
				$item->childs = null;

				EasyBlogHelper::buildNestedCategories($item->id, $item);

				$catIds     = array();
				$catIds[]   = $item->id;
				EasyBlogHelper::accessNestedCategoriesId($item, $catIds);

				$excludeCats    = array_merge($excludeCats, $catIds);
			}
		}

		return $excludeCats;
	}

	public static function getViewableTeamIds($userId = '')
	{

		$db 			= JFactory::getDBO();
		$my             = '';

		if( empty($userId) )
		{
			$my 			= JFactory::getUser();
		}
		else
		{
			$my 			= JFactory::getUser($userId);
		}

		$teamBlogIds    = '';


		if( $my->id == 0)
		{
			//get team id with access == 3
			$query  = 'select `id` FROM `#__easyblog_team` where `access` = ' . $db->Quote( '3' );
			$query  .= ' and `published` = ' . $db->Quote( '1' );
			$db->setQuery($query);

			$result 		= $db->loadResultArray();
			return $result;
		}
		else
		{

			$gid	= EasyBlogHelper::getUserGids( $userId );
			$gids   = '';
			if( count( $gid ) > 0 )
			{
				foreach( $gid as $id)
				{
					$gids   .= ( empty($gids) ) ? $db->Quote( $id ) : ',' . $db->Quote( $id );
				}
			}

			// get the teamid from this user.
			$query  = 'select distinct `id` from `#__easyblog_team` as t left join `#__easyblog_team_users` as tu on t.id = tu.team_id';
			$query  .= ' left join `#__easyblog_team_groups` as tg on t.id = tg.team_id';
			$query  .= ' where t.`published` = ' . $db->Quote( '1' );
			$query  .= ' and (tu.`user_id` = ' . $db->Quote( $my->id );
			$query  .= ' OR t.`access` = ' . $db->Quote( '3' );
			$query  .= ' OR tg.group_id IN (' . $gids . ')';
			$query  .= ')';

			// echo $query;exit;

			$db->setQuery($query);

			$result 		= $db->loadResultArray();
			return $result;
		}

	}

	public static function getDefaultCategoryId()
	{
		$db = JFactory::getDBO();

		$gid	= EasyBlogHelper::getUserGids();
		$gids   = '';
		if( count( $gid ) > 0 )
		{
			foreach( $gid as $id)
			{
				$gids   .= ( empty($gids) ) ? $db->Quote( $id ) : ',' . $db->Quote( $id );
			}
		}

		$query  = 'SELECT a.id';
		$query  .= ' FROM `#__easyblog_category` AS a';
		$query  .= ' WHERE a.`published` = ' . $db->Quote( '1' );
		$query  .= ' AND a.`default` = ' . $db->Quote( '1' );
		$query 	.= ' and a.id not in (';
		$query 	.= ' 	select id from `#__easyblog_category` as c';
		$query 	.= ' 	where not exists (';
		$query 	.= '			select b.category_id from `#__easyblog_category_acl` as b';
		$query 	.= '				where b.category_id = c.id and b.`acl_id` = '. $db->Quote( CATEGORY_ACL_ACTION_SELECT );
		$query 	.= '				and b.type = ' . $db->Quote('group');
		$query 	.= '				and b.content_id IN (' . $gids . ')';
		$query 	.= '      	)';
		$query 	.= ' 	and c.`private` = ' . $db->Quote( CATEGORY_PRIVACY_ACL );
		$query 	.= '	)';
		$query  .= ' AND a.`parent_id` NOT IN (SELECT `id` FROM `#__easyblog_category` AS e WHERE e.`published` = ' . $db->Quote( '0' ) . ' AND e.`parent_id` = ' . $db->Quote( '0' ) . ' )';
		$query  .= ' ORDER BY a.`lft` LIMIT 1';

		$db->setQuery( $query );
		$result = $db->loadResult();

		return ( empty( $result ) ) ? '0' : $result ;
	}

	public static function isBlogger( $userId )
	{
		if( empty( $userId ) )
			return false;

		$acl = EasyBlogACLHelper::getRuleSet( $userId );
		if( $acl->rules->add_entry )
		{
			return true;
		}
		else
		{
			return false;
		}

	}

	public static function getUniqueFileName($originalFilename, $path)
	{
		$ext			= JFile::getExt($originalFilename);
		$ext			= $ext ? '.'.$ext : '';
		$uniqueFilename	= JFile::stripExt($originalFilename);

		$i = 1;

		while( JFile::exists($path.DS.$uniqueFilename.$ext) )
		{
			// $uniqueFilename	= JFile::stripExt($originalFilename) . '-' . $i;
			$uniqueFilename	= JFile::stripExt($originalFilename) . '_' . $i . '_' . JFactory::getDate()->toFormat( "%Y%m%d-%H%M%S" );
			$i++;
		}

		//remove the space into '-'
		$uniqueFilename = str_ireplace(' ', '-', $uniqueFilename);

		return $uniqueFilename.$ext;
	}

	public static function getCategoryMenuBloggerId()
	{
		$itemId			= JRequest::getInt('Itemid', 0);
		$menu			= JFactory::getApplication()->getMenu();
		$menuparams		= $menu->getParams( $itemId );
		$catBloggerId	= $menuparams->get('cat_bloggerid', '');

		return $catBloggerId;
	}

	// Add canonical URL to satify Googlebot. Incase they think it's duplicated content.
	public static function addCanonicalURL( $extraFishes = array() )
	{
		if (empty( $extraFishes ))
		{
			return;
		}

		$juri		= JURI::getInstance();

		foreach( $extraFishes as $fish )
		{
			$juri->delVar( $fish );
		}

		$preferredURL	= $juri->toString();

		$document	= JFactory::getDocument();
		$document->addHeadLink( $preferredURL, 'canonical', 'rel');
	}
}
